import "./interop";
import * as pcl from "./pcl";

let minLogLevel = 0;
let pathTransformer = null;

export const ECL_State = {
  get RootObject() {
    return ECL_GetRootObject();
  },
  get CurrentObject() {
    return ECL_GetCurrentObject();
  },
  get Context() {
    return ECL_GetContext();
  },
  get SecurityContext() {
    return ECL_GetSecurityContext();
  },
  get Store() {
    return ECL_GetStore();
  },
  get Bom() {
    return ECL_GetBom();
  },
};

export const ECL_LogLevel = {
  get Trace() {
    return 0;
  },
  get Debug() {
    return 1;
  },
  get Information() {
    return 2;
  },
  get Warning() {
    return 3;
  },
  get Error() {
    return 4;
  },
  get Critical() {
    return 5;
  },
  get None() {
    return 6;
  },
};

export function ECL_SetMinimumLogLevel(level) {
  minLogLevel = NumberOrZero(level);
}

export function ECL_GetMinimumLogLevel() {
  return minLogLevel;
}

export function ECL_LogTrace(message) {
  if (minLogLevel <= ECL_LogLevel.Trace) {
    return pcl.LogTrace(StringOrNull(message));
  }
}

export function ECL_LogDebug(message) {
  if (minLogLevel <= ECL_LogLevel.Debug) {
    return pcl.LogDebug(StringOrNull(message));
  }
}

export function ECL_LogInformation(message) {
  if (minLogLevel <= ECL_LogLevel.Information) {
    return pcl.LogInformation(StringOrNull(message));
  }
}

export function ECL_LogWarning(message) {
  if (minLogLevel <= ECL_LogLevel.Warning) {
    return pcl.LogWarning(StringOrNull(message));
  }
}

export function ECL_LogError(message) {
  if (minLogLevel <= ECL_LogLevel.Error) {
    return pcl.LogError(StringOrNull(message));
  }
}

export function ECL_LogCritical(message) {
  if (minLogLevel <= ECL_LogLevel.Critical) {
    return pcl.LogCritical(StringOrNull(message));
  }
}

export function ECL_Trace(message) {
  return ECL_LogInformation(message);
}

export function ECL_GetContext() {
  return pcl.GetContext();
}

export function ECL_GetContextValue(key) {
  return ValueOrEmpty(pcl.GetContextValue(StringOrNull(key)));
}

export function ECL_GetSecurityContext() {
  return pcl.GetSecurityContext();
}

export function ECL_GetSecurityContextValue(key) {
  return ValueOrEmpty(pcl.GetSecurityContextValue(StringOrNull(key)));
}

export function ECL_GetStore() {
  return pcl.GetStore();
}

export function ECL_GetStoreValue(key) {
  return ValueOrEmpty(pcl.GetStoreValue(StringOrNull(key)));
}

export function ECL_SetStoreValue(key, value) {
  return pcl.SetStoreValue(StringOrNull(key), PrimitiveOrNull(value));
}

export function ECL_RemoveStoreValue(key) {
  return pcl.RemoveStoreValue(StringOrNull(key));
}

export function ECL_ClearStore() {
  return pcl.ClearStore();
}

export function ECL_GetUserVariable(key) {
  return ValueOrEmpty(pcl.GetStoreValue(StringOrNull(key)));
}

export function ECL_SetUserVariable(key, value) {
  return pcl.SetStoreValue(StringOrNull(key), PrimitiveOrNull(value));
}

export function ECL_DeleteUserVariable(key) {
  return pcl.RemoveStoreValue(StringOrNull(key));
}

export function ECL_DeleteAllUserVariables() {
  return pcl.ClearStore();
}

export function ECL_GetInfoVariable(key) {
  return ValueOrEmpty(pcl.GetVariableValue(StringOrNull(key)));
}

export function ECL_SetInfoVariable(key, value) {
  return pcl.SetVariableValue(StringOrNull(key), PrimitiveOrNull(value));
}

export function ECL_DeleteInfoVariable(key) {
  return pcl.RemoveVariableValue(StringOrNull(key));
}

export function ECL_DeleteAllInfoVariables() {
  return pcl.ClearVariables();
}

export function ECL_GetCacheValue(key) {
  return ValueOrEmpty(pcl.GetCacheValue(StringOrNull(key)));
}

export function ECL_SetCacheValue(key, value) {
  return pcl.SetCacheValue(StringOrNull(key), StringOrNull(value));
}

export function ECL_RemoveCacheValue(key) {
  return pcl.RemoveCacheValue(StringOrNull(key));
}

export function ECL_SetBomHierarchical(hierarchical) {
  return pcl.SetBomHierarchical(BooleanOrFalse(hierarchical));
}

export function ECL_SetBomObjectHierarchical(objectPath, hierarchical) {
  return pcl.SetBomObjectHierarchical(StringOrEmpty(objectPath), BooleanOrFalse(hierarchical));
}

export function ECL_IsBomHierarchical() {
  return pcl.IsBomHierarchical();
}

export function ECL_AddBomItem(partObject) {
  return pcl.AddBomItem(partObject);
}

export function ECL_GetBom() {
  return pcl.GetBom();
}

export function ECL_GetUpdatedBom(bomObject) {
  return pcl.GetUpdatedBom(bomObject);
}

export function ECL_SetCustomBom(bomObject) {
  return pcl.SetCustomBom(bomObject);
}

export function ECL_IsConditionTrue(ruleId) {
  return pcl.IsConditionTrue(StringOrNull(ruleId));
}

export function ECL_GetTrueConditions() {
  return pcl.GetTrueConditions();
}

export function ECL_GetValueTypecode(attrPath, valueId) {
  return ValueOrEmpty(pcl.GetValueTypeCodeFragment(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId)));
}

export function ECL_SetValueTypecode(attrPath, valueId, typeCode) {
  return pcl.SetValueTypeCodeFragment(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId), StringOrNull(typeCode));
}

export function ECL_GetAttributeTypecode(attrPath) {
  return ValueOrEmpty(pcl.GetAttributeTypeCode(ECL_ConvertLegacyPath(attrPath)));
}

export function ECL_GetCfgTypecodeScheme(objectPath) {
  return ValueOrEmpty(pcl.GetObjectTypeCodeTemplate(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_GetTypecode() {
  return ValueOrEmpty(pcl.GetCurrentObjectTypeCode());
}

export function ECL_GetRootObjectTypeCode() {
  return ValueOrEmpty(pcl.GetRootObjectTypeCode());
}

export function ECL_SetTypecode(typeCode) {
  return pcl.SetCurrentObjectTypeCode(StringOrNull(typeCode));
}

export function ECL_UpdateRootObjectTypeCode() {
  return ValueOrEmpty(pcl.UpdateRootObjectTypeCode());
}

export function ECL_UpdateCurrentObjectTypeCode() {
  return ValueOrEmpty(pcl.UpdateCurrentObjectTypeCode());
}

export function ECL_UpdateObjectTypeCode(objectPath) {
  return ValueOrEmpty(pcl.UpdateObjectTypeCode(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_TranslateValue(attrId, valueId) {
  return pcl.GetTranslationForContext(StringOrNull(valueId), StringOrNull(attrId));
}

export function ECL_TranslateAttributeValue(attrId, valueId) {
  return pcl.GetTranslationForContext(StringOrNull(valueId), StringOrNull(attrId));
}

export function ECL_Translate(id, language) {
  if (language) {
    return pcl.GetTranslationForLanguage(StringOrNull(id), StringOrNull(language));
  }

  return pcl.GetTranslation(StringOrNull(id));
}

export function ECL_AttributeN(attributeId) {
  return attributeId.startsWith("A_") ? attributeId.substring(2) : attributeId;
}

export function ECL_ValueN(valueId) {
  return valueId.startsWith("V_") ? valueId.substring(2) : valueId;
}

export function ECL_GetRootObject() {
  return pcl.GetRootObject();
}

export function ECL_GetCurrentObject() {
  return pcl.GetCurrentObject();
}

export function ECL_GetCurrentLanguage() {
  return pcl.GetLocaleId().substring(0, 2).toUpperCase();
}

export function ECL_SetPathTransformer(transformer) {
  if (typeof transformer === "function") {
    pathTransformer = transformer;

    return true;
  }

  pathTransformer = null;

  return false;
}

export function ECL_GetPathTransformer() {
  return pathTransformer;
}

export function ECL_RemovePathTransformer() {
  pathTransformer = null;

  return true;
}

export function ECL_ConvertLegacyPath(rawPath) {
  const pathBase = StringOrNull(rawPath);
  const path = pathTransformer ? pathTransformer(pathBase) : pathBase;

  if (!path) {
    return "";
  }

  let elements = [];
  let isAbsolute = false;

  for (const element of path.split(/[.[\]]/)) {
    switch (element) {
      case "":
      case "RANGE":
      case "SEQUENCE":
        break;
      case "ROOT":
        elements = [];
        isAbsolute = true;
        break;
      case "OWNER":
      case "PARENT":
        elements.push("..");
        elements.push("..");
        break;
      case "OWNERATTR":
        elements.push("..");
        break;
      default:
        elements.push(element);
        break;
    }
  }

  const newPath = elements.join("/");

  if (isAbsolute) {
    return "/" + newPath;
  }

  return newPath;
}

export function ECL_GetCurrentCfgStructurePath() {
  const rootObject = ECL_GetRootCfgObject();
  const currentObject = StringOrEmpty(pcl.GetCurrentObjectPath());

  if (currentObject === "/") {
    return rootObject;
  }

  const path = rootObject + currentObject;
  const legacyPath = [];
  let count = 0;

  for (const element of path.split("/")) {
    if (count === 0) {
      legacyPath.push(element);
    } else if (count % 2 === 0) {
      legacyPath.push("RANGE");
      legacyPath.push(element);
    } else {
      legacyPath.push(element);
    }

    ++count;
  }

  return legacyPath.join(".");
}

export function ECL_GetCurrentCfgStructureNumber() {
  const path = StringOrEmpty(pcl.GetCurrentObjectPath()).split("/");
  const index = NumberOrZero(path[path.length - 1]);

  return index + 1;
}

export function ECL_IsRootObjectComplete() {
  return pcl.IsRootObjectComplete();
}

export function ECL_IsCurrentObjectComplete() {
  return pcl.IsCurrentObjectComplete();
}

export function ECL_IsObjectComplete(objectPath) {
  return pcl.IsObjectComplete(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetCfgStateComplete(objectPath) {
  return pcl.IsObjectComplete(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_IsAttributeComplete(attrPath) {
  return pcl.IsAttributeComplete(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_UpdateRootObjectCompleteness() {
  return pcl.UpdateRootObjectCompleteness();
}

export function ECL_UpdateCurrentObjectCompleteness() {
  return pcl.UpdateCurrentObjectCompleteness();
}

export function ECL_UpdateObjectCompleteness(objectPath) {
  return pcl.UpdateObjectCompleteness(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetRootCfgObject() {
  return pcl.GetRootObjectId();
}

export function ECL_GetCurrentCfgObject() {
  return pcl.GetCurrentObjectId();
}

export function ECL_GetCurrentCFGObject() {
  return pcl.GetCurrentObjectId();
}

export function ECL_ObjectValid(objectPath) {
  return pcl.IsObject(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetObjectPath(objectPath) {
  return ValueOrEmpty(pcl.GetObjectPath(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_GetObjectId(objectPath) {
  return ValueOrEmpty(pcl.GetObjectId(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_GetObjectGuid(objectPath) {
  return ValueOrEmpty(pcl.GetObjectGuid(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_GetCfgObjectGuid(objectPath) {
  return ValueOrEmpty(pcl.GetObjectGuid(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_GetCfgObjectPath(objectPath) {
  return ValueOrEmpty(pcl.GetObjectPath(ECL_ConvertLegacyPath(objectPath)));
}

export function ECL_AttributeValid(attrPath) {
  return pcl.IsAttribute(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAttributePath(attrPath) {
  return ValueOrEmpty(pcl.GetAttributePath(ECL_ConvertLegacyPath(attrPath)));
}

export function ECL_GetAttributeId(attrPath) {
  return ValueOrEmpty(pcl.GetAttributeId(ECL_ConvertLegacyPath(attrPath)));
}

export function ECL_GetAttributeDataType(attrPath) {
  return ValueOrEmpty(pcl.GetAttributeDataType(ECL_ConvertLegacyPath(attrPath))).toUpperCase();
}

export function ECL_GetObjectType(path) {
  if (ECL_ObjectValid(path)) {
    return 1;
  }

  if (ECL_AttributeValid(path)) {
    return 2;
  }

  return 3;
}

export function ECL_GetCurrentAttribute() {
  if (pcl.IsCurrentObjectFirstRun()) {
    return "@INIT@";
  }

  return pcl.GetCurrentAttribute();
}

export function ECL_IsRootObjectFirstRun() {
  return pcl.IsRootObjectFirstRun();
}

export function ECL_IsCurrentObjectFirstRun() {
  return pcl.IsCurrentObjectFirstRun();
}

export function ECL_GetModifiedAttribute() {
  return pcl.GetCurrentAttribute();
}

export function ECL_ActivateAttributes(objectPath) {
  return pcl.SetAttributesVisible(ECL_ConvertLegacyPath(objectPath), true);
}

export function ECL_DeactivateAttributes(objectPath) {
  return pcl.SetAttributesVisible(ECL_ConvertLegacyPath(objectPath), false);
}

export function ECL_EnableAttributes(objectPath) {
  return pcl.SetAttributesAllowed(ECL_ConvertLegacyPath(objectPath), true);
}

export function ECL_DisableAttributes(objectPath) {
  return pcl.SetAttributesAllowed(ECL_ConvertLegacyPath(objectPath), false);
}

export function ECL_GetAllAttributes(objectPath) {
  return pcl.GetAttributes(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetPreviousAttributes(attributePath) {
  return pcl.GetPreviousAttributes(ECL_ConvertLegacyPath(attributePath));
}

export function ECL_GetNextAttributes(attributePath) {
  return pcl.GetNextAttributes(ECL_ConvertLegacyPath(attributePath));
}

export function ECL_GetAllActiveAttributes(objectPath) {
  return pcl.GetVisibleAttributes(ECL_ConvertLegacyPath(objectPath), true);
}

export function ECL_GetAllEnabledAttributes(objectPath) {
  return pcl.GetAllowedAttributes(ECL_ConvertLegacyPath(objectPath), true);
}

export function ECL_GetAllUserChangedAttributes(objectPath) {
  return pcl.GetUserSelectedAttributes(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetAllAttributeValues(attrPath) {
  return pcl.GetValues(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAllDefaultAttributeValues(attrPath) {
  return pcl.GetDefaultValues(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAllSelectedAttributeValues(attrPath) {
  return pcl.GetSelectedValues(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAllActiveAttributeValues(attrPath) {
  return pcl.GetActiveValues(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_GetAllEnabledAttributeValues(attrPath) {
  return pcl.GetAllowedValues(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_ResetAttribute(attrPath) {
  return pcl.ClearAttribute(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_ResetBelowAttribute(attrPath) {
  return pcl.ClearNextAttributes(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_ResetBelowCurrentAttribute() {
  return pcl.ClearNextAttributes(null);
}

export function ECL_DeselectAttribute(attrPath) {
  return pcl.DeselectValues(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_DeselectAttributeDefault(attrPath) {
  return pcl.DeselectDefaultValues(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_IsSubConfiguration(attrPath) {
  return pcl.IsSubConfiguration(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetCard(attrPath) {
  return pcl.GetSubConfigurationCount(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetCardMax(attrPath) {
  return pcl.GetSubConfigurationMax(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetCardMin(attrPath) {
  return pcl.GetSubConfigurationMin(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_SetCardMax(attrPath, cardMax) {
  return pcl.SetSubConfigurationMax(ECL_ConvertLegacyPath(attrPath), NumberOrZero(cardMax));
}

export function ECL_SetCardMin(attrPath, cardMin) {
  return pcl.SetSubConfigurationMin(ECL_ConvertLegacyPath(attrPath), NumberOrZero(cardMin));
}

export function ECL_ResetCard(attrPath) {
  return pcl.ResetSubConfigurationMinMax(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_SetValue(attrPath, valueId) {
  if (valueId === null || valueId === undefined || valueId === "") {
    return ECL_DeselectAttribute(attrPath);
  }

  return pcl.SetValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_GetValue(attrPath) {
  return ValueOrEmpty(pcl.GetSelectedValue(ECL_ConvertLegacyPath(attrPath)));
}

export function ECL_GetValueCount(attrPath) {
  return pcl.GetValueCount(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetValueIndex(attrPath, valueId) {
  return pcl.GetValueIndex(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetValueByIndex(attrPath, index) {
  return pcl.GetValueByIndex(ECL_ConvertLegacyPath(attrPath), NumberOrMinusOne(index));
}

export function ECL_GetValueImage(attrPath, valueId) {
  return ValueOrEmpty(pcl.GetValueImage(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId)));
}

export function ECL_SetValuePartClass(attrPath, valueId, partClass) {
  return pcl.SetValuePartClass(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId), StringOrNull(partClass));
}

export function ECL_GetValuePartClass(attrPath, valueId) {
  return ValueOrEmpty(pcl.GetValuePartClass(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId)));
}

export function ECL_SetValueMass(attrPath, valueId, mass) {
  return pcl.SetValueMass(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId), NumberOrNull(mass));
}

export function ECL_GetValueMass(attrPath, valueId) {
  return NumberOrZero(pcl.GetValueMass(ECL_ConvertLegacyPath(attrPath), PrimitiveOrNull(valueId)));
}

export function ECL_SetValueImage(attrPath, valueId, imageUrl) {
  return pcl.SetValueImage(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), StringOrNull(imageUrl));
}

export function ECL_SetValueImageDefault(attrPath, valueId) {
  return pcl.ResetValueImage(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_ValueSelected(attrPath, valueId) {
  return pcl.IsValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_ValueSelectedAnywhere(attrId, valueId) {
  return pcl.IsValueSelectedAnywhere(ECL_ConvertLegacyPath(attrId), StringOrNull(valueId));
}

export function ECL_SelectValueEverywhere(attrId, valueId) {
  return pcl.SelectValueEverywhere(StringOrNull(attrId), PrimitiveOrNull(valueId));
}

export function ECL_DeselectValueEverywhere(attrId, valueId) {
  return pcl.DeselectValueEverywhere(StringOrNull(attrId), PrimitiveOrNull(valueId));
}

export function ECL_DeselectValuesEverywhere(attrId) {
  return pcl.DeselectValuesEverywhere(StringOrNull(attrId));
}

export function ECL_SetValueSelectedEverywhere(attrId, valueId, selected) {
  return pcl.SetValueSelectedEverywhere(StringOrNull(attrId), PrimitiveOrNull(valueId), BooleanOrTrue(selected));
}

export function ECL_SelectValue(attrPath, valueId) {
  if (valueId === null || valueId === undefined || valueId === "") {
    return ECL_DeselectAttribute(attrPath);
  }

  return pcl.SetValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_SelectValueSafe(attrPath, valueId) {
  if (valueId === null || valueId === undefined || valueId === "") {
    return ECL_DeselectAttribute(attrPath);
  }

  return pcl.SetValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_DeselectValue(attrPath, valueId) {
  return pcl.SetValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), false);
}

export function ECL_ValueDefault(attrPath, valueId) {
  return pcl.IsValueDefault(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_SelectValueDefault(attrPath, valueId) {
  return pcl.SetDefaultValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_DeselectValueDefault(attrPath, valueId) {
  return pcl.SetDefaultValueSelected(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), false);
}

export function ECL_SetValueQuantity(attrPath, valueId, quantity) {
  return pcl.SetValueQuantity(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), NumberOrNull(quantity));
}

export function ECL_SetValueDefaultQuantity(attrPath, valueId) {
  return pcl.ResetValueQuantity(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetValueDefaultQuantity(attrPath, valueId) {
  return pcl.GetValueQuantityDefault(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetValueQuantity(attrPath, valueId) {
  return pcl.GetValueQuantity(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_DisableValueQuantity(attrPath, valueId) {
  return pcl.SetValueQuantity(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), null);
}

export function ECL_EnableValueQuantity(attrPath, valueId) {
  const path = ECL_ConvertLegacyPath(attrPath);
  const quantity = NumberOrOne(pcl.GetValueQuantityDefault(path, StringOrNull(valueId)));

  return pcl.SetValueQuantity(path, StringOrNull(valueId), quantity);
}

export function ECL_SetPrice(attrPath, valueId, price) {
  return pcl.SetValuePrice(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), NumberOrNull(price));
}

export function ECL_SetPriceDefault(attrPath, valueId) {
  return pcl.ResetValuePrice(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetPrice(attrPath, valueId) {
  return pcl.GetValuePrice(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetPriceDefault(attrPath, valueId) {
  return pcl.GetValuePriceDefault(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetBasePrice(attrPath, valueId) {
  return pcl.GetValuePriceDefault(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_GetAttributeCount(objectPath) {
  return pcl.GetAttributeCount(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetAttributeIndex(attrPath) {
  return pcl.GetAttributeIndex(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAttributeByIndex(objectPath, index) {
  return pcl.GetAttributeByIndex(ECL_ConvertLegacyPath(objectPath), NumberOrMinusOne(index));
}

export function ECL_GetAttributeMV(attrPath) {
  return pcl.IsAttributeMultiValued(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeMultiValued(attrPath) {
  return pcl.IsAttributeMultiValued(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeVisible(attrPath) {
  return pcl.IsAttributeVisible(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeReadOnly(attrPath) {
  return pcl.IsAttributeReadOnly(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeEditable(attrPath) {
  return pcl.IsAttributeEditable(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeAutoSelected(attrPath) {
  return pcl.IsAttributeAutoSelected(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeActive(attrPath) {
  return pcl.IsAttributeVisible(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeEnabled(attrPath) {
  return pcl.IsAttributeAllowed(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeUserChanged(attrPath) {
  return pcl.IsAttributeUserSelected(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_AttributeRequired(attrPath) {
  return pcl.IsAttributeRequired(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_RequireAttribute(attrPath) {
  return pcl.SetAttributeRequired(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_UnrequireAttribute(attrPath) {
  return pcl.SetAttributeRequired(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_AtLeastOneSelected(attrPath) {
  return pcl.IsAttributeAssigned(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_IsAssigned(attrPath) {
  return pcl.IsAttributeAssigned(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_IsAttributeLDP(attrPath) {
  return pcl.IsAttributeLanguageDependent(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_ValueValid(attrPath, valueId) {
  return pcl.IsValue(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_ValueActive(attrPath, valueId) {
  return pcl.IsValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_ValueEnabled(attrPath, valueId) {
  return pcl.IsValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId)) && pcl.IsValueAllowed(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId));
}

export function ECL_ActivateAllAttributeValues(attrPath) {
  return pcl.SetValuesActive(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_DeactivateAllAttributeValues(attrPath) {
  return pcl.SetValuesActive(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_EnableAllAttributeValues(attrPath) {
  return pcl.SetValuesAllowed(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_DisableAllAttributeValues(attrPath) {
  return pcl.SetValuesAllowed(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_ActivateValue(attrPath, valueId) {
  return pcl.SetValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_ShowValue(attrPath, valueId) {
  return pcl.SetValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_DeactivateValue(attrPath, valueId) {
  return pcl.SetValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), false);
}

export function ECL_HideValue(attrPath, valueId) {
  return pcl.SetValueActive(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), false);
}

export function ECL_EnableValue(attrPath, valueId) {
  return pcl.SetValueAllowed(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), true);
}

export function ECL_DisableValue(attrPath, valueId) {
  return pcl.SetValueAllowed(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), false);
}

export function ECL_GetValueArticleNumber(attrPath, valueId) {
  return StringOrEmpty(pcl.GetValuePartId(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId)));
}

export function ECL_SetValueArticleNumber(attrPath, valueId, partId) {
  return pcl.SetValuePartId(ECL_ConvertLegacyPath(attrPath), StringOrNull(valueId), StringOrNull(partId));
}

export function ECL_TranslateAttribute(attrPath) {
  return pcl.GetTranslation(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_EnableAttribute(attrPath) {
  return pcl.SetAttributeAllowed(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_DisableAttribute(attrPath) {
  return pcl.SetAttributeAllowed(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_ActivateAttribute(attrPath) {
  return pcl.SetAttributeVisible(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_ShowAttribute(attrPath) {
  return pcl.SetAttributeVisible(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_DeactivateAttribute(attrPath) {
  return pcl.SetAttributeVisible(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_HideAttribute(attrPath) {
  return pcl.SetAttributeVisible(ECL_ConvertLegacyPath(attrPath), false);
}

export function ECL_ResetAttributeGroup(groupPath) {
  return pcl.ClearGroup(ECL_ConvertLegacyPath(groupPath));
}

export function ECL_EnableAttributeGroup(groupPath) {
  return pcl.SetGroupAllowed(ECL_ConvertLegacyPath(groupPath), true);
}

export function ECL_DisableAttributeGroup(groupPath) {
  return pcl.SetGroupAllowed(ECL_ConvertLegacyPath(groupPath), false);
}

export function ECL_ActivateAttributeGroup(groupPath) {
  return pcl.SetGroupVisible(ECL_ConvertLegacyPath(groupPath), true);
}

export function ECL_DeactivateAttributeGroup(groupPath) {
  return pcl.SetGroupVisible(ECL_ConvertLegacyPath(groupPath), false);
}

export function ECL_GetAttributeGroupAttributes(groupPath) {
  return pcl.GetGroupAttributes(ECL_ConvertLegacyPath(groupPath));
}

export function ECL_GetAllAttributeGroups(objectPath) {
  return pcl.GetGroups(ECL_ConvertLegacyPath(objectPath));
}

export function ECL_GetMin(attrPath) {
  return pcl.GetAttributeMin(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetMax(attrPath) {
  return pcl.GetAttributeMax(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_SetMinMax(attrPath, minValue, maxValue) {
  return pcl.SetAttributeMinMax(ECL_ConvertLegacyPath(attrPath), StringOrNull(minValue), StringOrNull(maxValue));
}

export function ECL_ResetMinMax(attrPath) {
  return pcl.ResetAttributeMinMax(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_CreateSubstructure(attrPath) {
  return pcl.AddSubConfiguration(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_RunSubstructures(attrPath) {
  return pcl.RunSubConfigurations(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_ResetSubstructures(attrPath) {
  return pcl.ResetSubConfigurations(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_CopySubstructure(attrPath, index1, index2) {
  return pcl.CopySubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index1), NumberOrZero(index2 ?? index1 + 1));
}

export function ECL_MoveSubstructure(attrPath, index1, index2) {
  return pcl.MoveSubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index1), NumberOrZero(index2 ?? index1 + 1));
}

export function ECL_DeleteSubstructure(attrPath, index) {
  return pcl.RemoveSubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index));
}

export function ECL_InsertSubstructure(attrPath, index) {
  return pcl.InsertSubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index));
}

export function ECL_ResetSubstructure(attrPath, index) {
  return pcl.ResetSubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index));
}

export function ECL_RunSubstructure(attrPath, index) {
  return pcl.RunSubConfiguration(ECL_ConvertLegacyPath(attrPath), NumberOrZero(index));
}

export function ECL_OpenSubstructure(attrPath, index, option) {
  index = NumberOrZero(index);

  if (option === "create") {
    const count = ECL_GetCard(attrPath);

    if (index >= count) {
      for (let i = 0; i < index - count + 1; ++i) {
        ECL_CreateSubstructure(attrPath);
      }
    }
  }

  return pcl.OpenSubConfiguration(ECL_ConvertLegacyPath(attrPath), index);
}

export function ECL_CloseSubstructure() {
  return pcl.CloseSubConfiguration();
}

export function ECL_ResetObject(objPath) {
  return pcl.ResetObject(ECL_ConvertLegacyPath(objPath));
}

export function ECL_SetCfgObjectQuantity(objPath, quantity) {
  return pcl.SetObjectQuantity(ECL_ConvertLegacyPath(objPath), NumberOrNull(quantity));
}

export function ECL_GetCfgObjectQuantity(objPath) {
  return pcl.GetObjectQuantity(ECL_ConvertLegacyPath(objPath));
}

export function ECL_GetObjectProperties(objPath) {
  return pcl.GetObjectProperties(ECL_ConvertLegacyPath(objPath));
}

export function ECL_GetObjectPropertyCount(objPath) {
  return pcl.GetObjectPropertyCount(ECL_ConvertLegacyPath(objPath));
}

export function ECL_GetObjectPropertyByIndex(objPath, index) {
  return ValueOrEmpty(pcl.GetObjectPropertyByIndex(ECL_ConvertLegacyPath(objPath), NumberOrMinusOne(index)));
}

export function ECL_GetObjectPropertyIdByIndex(objPath, index) {
  return ValueOrEmpty(pcl.GetObjectPropertyIdByIndex(ECL_ConvertLegacyPath(objPath), NumberOrMinusOne(index)));
}

export function ECL_GetObjectProperty(objPath, propertyId) {
  return ValueOrEmpty(pcl.GetObjectPropertyValue(ECL_ConvertLegacyPath(objPath), StringOrEmpty(propertyId)));
}

export function ECL_ResetObjectProperty(objPath, propertyId) {
  return pcl.ResetObjectProperty(ECL_ConvertLegacyPath(objPath), StringOrEmpty(propertyId));
}

export function ECL_SetObjectProperty(objPath, propertyId, propertyValue) {
  return pcl.SetObjectPropertyValue(ECL_ConvertLegacyPath(objPath), StringOrEmpty(propertyId), StringOrEmpty(propertyValue));
}

export function ECL_GetAttributeProperties(attrPath) {
  return pcl.GetAttributeProperties(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAttributePropertyCount(attrPath) {
  return pcl.GetAttributePropertyCount(ECL_ConvertLegacyPath(attrPath));
}

export function ECL_GetAttributePropertyByIndex(attrPath, index) {
  return ValueOrEmpty(pcl.GetAttributePropertyByIndex(ECL_ConvertLegacyPath(attrPath), NumberOrMinusOne(index)));
}

export function ECL_GetAttributePropertyIdByIndex(attrPath, index) {
  return ValueOrEmpty(pcl.GetAttributePropertyIdByIndex(ECL_ConvertLegacyPath(attrPath), NumberOrMinusOne(index)));
}

export function ECL_GetAttributeProperty(attrPath, propertyId) {
  return ValueOrEmpty(pcl.GetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(propertyId)));
}

export function ECL_ResetAttributeProperty(attrPath, propertyId) {
  return pcl.ResetAttributeProperty(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(propertyId));
}

export function ECL_SetAttributeProperty(attrPath, propertyId, propertyValue) {
  return pcl.SetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(propertyId), StringOrEmpty(propertyValue));
}

export function ECL_GetValueProperties(attrPath, valueId) {
  return pcl.GetValueProperties(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId));
}

export function ECL_GetValuePropertyCount(attrPath, valueId) {
  return pcl.GetValuePropertyCount(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId));
}

export function ECL_GetValuePropertyByIndex(attrPath, valueId, index) {
  return ValueOrEmpty(pcl.GetValuePropertyByIndex(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), NumberOrMinusOne(index)));
}

export function ECL_GetValuePropertyIdByIndex(attrPath, valueId, index) {
  return ValueOrEmpty(pcl.GetValuePropertyIdByIndex(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), NumberOrMinusOne(index)));
}

export function ECL_GetValueProperty(attrPath, valueId, propertyId) {
  return ValueOrEmpty(pcl.GetValuePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(propertyId)));
}

export function ECL_ResetValueProperty(attrPath, valueId, propertyId) {
  return pcl.ResetValueProperty(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(propertyId));
}

export function ECL_SetValueProperty(attrPath, valueId, propertyId, propertyValue) {
  return pcl.SetValuePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(propertyId), StringOrEmpty(propertyValue));
}

export function ECL_GetCfgObjectFlag(objPath, flagId) {
  return ValueOrEmpty(pcl.GetObjectPropertyValue(ECL_ConvertLegacyPath(objPath), StringOrEmpty(flagId)));
}

export function ECL_SetCfgObjectFlagDefault(objPath, flagId) {
  return pcl.ResetObjectProperty(ECL_ConvertLegacyPath(objPath), StringOrEmpty(flagId));
}

export function ECL_SetCfgObjectFlag(objPath, flagId, flagValue) {
  return pcl.SetObjectPropertyValue(ECL_ConvertLegacyPath(objPath), StringOrEmpty(flagId), StringOrEmpty(flagValue));
}

export function ECL_GetAttributeFlag(attrPath, flagId) {
  return ValueOrEmpty(pcl.GetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(flagId)));
}

export function ECL_SetAttributeFlagDefault(attrPath, flagId) {
  return pcl.ResetAttributeProperty(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(flagId));
}

export function ECL_SetAttributeFlag(attrPath, flagId, flagValue) {
  return pcl.SetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(flagId), StringOrEmpty(flagValue));
}

export function ECL_GetValueFlag(attrPath, valueId, flagId) {
  return ValueOrEmpty(pcl.GetValuePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(flagId)));
}

export function ECL_SetValueFlagDefault(attrPath, valueId, flagId) {
  return pcl.ResetValueProperty(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(flagId));
}

export function ECL_SetValueFlag(attrPath, valueId, flagId, flagValue) {
  return pcl.SetValuePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(flagId), StringOrEmpty(flagValue));
}

export function ECL_ModifyAttribute(attrPath) {
  return pcl.SetAttributeUserSelected(ECL_ConvertLegacyPath(attrPath), true);
}

export function ECL_SetAttributeHelpData(attrPath, helpDataType, helpData) {
  return pcl.SetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(helpDataType), StringOrEmpty(helpData));
}

export function ECL_SetValueHelpData(attrPath, valueId, helpDataType, helpData) {
  return pcl.SetValuePropertyValue(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), StringOrEmpty(helpDataType), StringOrEmpty(helpData));
}

export function ECL_GetCFGAttributeControl(attrPath) {
  return ValueOrEmpty(pcl.GetAttributePropertyValue(ECL_ConvertLegacyPath(attrPath), "Control"));
}

export function ECL_HighlightValue(attrPath, valueId, highlight) {
  return pcl.SetValueRecommendationScore(ECL_ConvertLegacyPath(attrPath), StringOrEmpty(valueId), highlight === "+" ? 1 : highlight === "-" ? -1 : 0);
}

export function ECL_HighlightAllAttributeValues(attrPath, highlight) {
  return pcl.SetValuesRecommendationScore(ECL_ConvertLegacyPath(attrPath), highlight === "+" ? 1 : highlight === "-" ? -1 : 0);
}

export function ECL_SelectValueQuantity(attrPath, valueId, quantity) {
  return ECL_SelectValue(attrPath, valueId) && ECL_SetValueQuantity(attrPath, valueId, quantity);
}

export function ECL_SelectValueQuantityDefault(attrPath, valueId) {
  return ECL_SelectValue(attrPath, valueId) && ECL_SetValueDefaultQuantity(attrPath, valueId);
}

export function ECL_SelectedAttributeValuesContain(attrPath, valueSubstring) {
  if (valueSubstring) {
    const values = ECL_GetAllSelectedAttributeValues(attrPath);

    for (let i = 0; i < values.length; ++i) {
      if (StringOrEmpty(values[i]).indexOf(valueSubstring) > -1) {
        return true;
      }
    }
  }

  return false;
}

export function ECL_ActivateAttributeAndSelectValue(attrPath, valueId) {
  let result = true;

  result &&= ECL_ActivateAttribute(attrPath);
  result &&= ECL_DeactivateAllAttributeValues(attrPath);
  result &&= ECL_ActivateValue(attrPath, valueId);
  result &&= ECL_SelectValue(attrPath, valueId);

  return result;
}

export function ECL_FreezeSingleValue(attrPath, valueId) {
  const values = StringOrEmpty(valueId).split("|");

  let result = true;

  result &&= ECL_DeactivateAllAttributeValues(attrPath);
  result &&= ECL_DisableAllAttributeValues(attrPath);

  for (let i = 0; i < values.length; ++i) {
    result &&= ECL_ActivateValue(attrPath, values[i]);
    result &&= ECL_EnableValue(attrPath, values[i]);
    result &&= ECL_SelectValue(attrPath, values[i]);
  }

  return result;
}

export function ECL_RestrictAttribute(attribute, values, options) {
  const inference = new VariantTables.Inference();
  const result = inference.restrictAttribute(attribute, values, options);

  return result;
}

export function ECL_RestrictAttributes(restrictions) {
  const inference = new VariantTables.Inference();
  const result = inference.restrictAttributes(restrictions);

  return result;
}

export function ECL_InferenceAttributes(database, tableName, options) {
  const simpleInference = new VariantTables.SimpleInference();
  const result = simpleInference.restrictAttributes(StringOrEmpty(database), StringOrEmpty(tableName), options);

  return result;
}

export function ECL_DatabaseExecuteQuery(database, statement, args) {
  return pcl.DatabaseExecuteQuery(StringOrEmpty(database), StringOrEmpty(statement), args);
}

export function ECL_DatabaseExecuteNonQuery(database, statement, args) {
  return pcl.DatabaseExecuteNonQuery(StringOrEmpty(database), StringOrEmpty(statement), args);
}

export function ECL_DatabaseGetTableColumns(database, tableName) {
  return pcl.DatabaseGetTableColumns(StringOrEmpty(database), StringOrEmpty(tableName));
}

export function ECL_ObjectFromDatabase(database, statement, args) {
  const result = pcl.DatabaseExecuteQuery(StringOrEmpty(database), StringOrEmpty(statement), args);

  return result.Rows;
}

export function ECL_GetLastDatabaseError(database) {
  return ValueOrEmpty(pcl.GetLastDatabaseError(StringOrEmpty(database)));
}

export function ECL_GetDatabaseProvider(database) {
  return ValueOrEmpty(pcl.GetDatabaseProvider(StringOrEmpty(database)));
}

export function ECL_GetControlType(attrPath) {
  const path = ECL_ConvertLegacyPath(attrPath);

  if (!pcl.IsAttribute(path)) {
    return "";
  }

  const controlType = pcl.GetAttributePropertyValue(path, "ControlType");

  if (controlType) {
    return controlType;
  }

  if (pcl.IsAttributeEditable(path)) {
    return "EDIT";
  }

  if (pcl.IsAttributeMultiValued(path)) {
    return "CHECK_H";
  }

  return "COMBO_H";
}

export function ECL_GetUserName() {
  ECL_LogError("ECL_GetUserName is not supported");

  return "GUEST";
}

export function ECL_GetUserGroup() {
  ECL_LogError("ECL_GetUserGroup is not supported");

  return "GUEST";
}

export function ECL_GetCurrentObjectId() {
  return pcl.GetCurrentObjectId();
}

export function ECL_GetCurrentObjectPath() {
  return pcl.GetCurrentObjectPath();
}

export function ECL_IsAttributeConfiguration(attrPath) {
  return BooleanOrFalse(pcl.IsAttributeConfiguration(ECL_ConvertLegacyPath(attrPath)));
}
