import "./interop";
import "./vt/VariantTables";
import * as pcl from "./ConfigurationApi.js";

export * from "./ConfigurationApi.js";

let minLogLevel = 0;

/**
 * @summary
 * Debug state. Only for debugging!
 */
export const State = {
  get RootObject() {
    return pcl.GetRootObject();
  },
  get CurrentObject() {
    return pcl.GetCurrentObject();
  },
  get Context() {
    return pcl.GetContext();
  },
  get SecurityContext() {
    return pcl.GetSecurityContext();
  },
  get Inputs() {
    return pcl.GetInputs();
  },
  get Outputs() {
    return pcl.GetOutputs();
  },
  get Variables() {
    return pcl.GetVariables();
  },
  get Store() {
    return pcl.GetStore();
  },
  get Bom() {
    return pcl.GetBom();
  }
};

/**
 * @summary
 * Log level constants.
 */
export const LogLevel = {
  get Trace() {
    return 0;
  },
  get Debug() {
    return 1;
  },
  get Information() {
    return 2;
  },
  get Warning() {
    return 3;
  },
  get Error() {
    return 4;
  },
  get Critical() {
    return 5;
  },
  get None() {
    return 6;
  },
};

/**
 * @summary
 * Sets the minimum log level for the configuration engine.
 *
 * @param {number} level
 */
export function SetMinimumLogLevel(level) {
  minLogLevel = NumberOrZero(level) | 0;
}

/**
 * @summary
 * Gets the minimum log level for the configuration engine.
 * The minimum log level can be set with SetMinimumLogLevel. The default value is 0.
 *
 * @returns {number} The minimum log level.
 */
export function GetMinimumLogLevel() {
  return minLogLevel;
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Trace".
 *
 * @param {string} message
 */
export function LogTrace(message) {
  if (minLogLevel <= LogLevel.Trace) {
    return pcl.LogTrace(message);
  }
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Debug".
 *
 * @param {string} message
 */
export function LogDebug(message) {
  if (minLogLevel <= LogLevel.Debug) {
    return pcl.LogDebug(message);
  }
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Information".
 *
 * @param {string} message
 */
export function LogInformation(message) {
  if (minLogLevel <= LogLevel.Information) {
    return pcl.LogInformation(message);
  }
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Warning".
 *
 * @param {string} message
 */
export function LogWarning(message) {
  if (minLogLevel <= LogLevel.Warning) {
    return pcl.LogWarning(message);
  }
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Error".
 *
 * @param {string} message
 */
export function LogError(message) {
  if (minLogLevel <= LogLevel.Error) {
    return pcl.LogError(message);
  }
}

/**
 * @summary
 * Logs a message if the current log level is less than or equal to the value for "Critical".
 *
 * @description If the current log level is greater, no action is taken.
 * The minimum log level can be set with SetMinimumLogLevel. The default value is 0.
 *
 * @param {string} message
 */
export function LogCritical(message) {
  if (minLogLevel <= LogLevel.Critical) {
    return pcl.LogCritical(message);
  }
}

/**
 * @summary
 * The function RestrictAttribute.
 *
 * @param {string} attribute
 * @param {array} values
 * @param {object} options
 *
 * @returns {object} The result set.
 */
export function RestrictAttribute(attribute, values, options) {
  const inference = new VariantTables.Inference();
  const result = inference.restrictAttribute(attribute, values, options);

  return result;
}

/**
 * @summary
 * The function RestrictAttributes.
 *
 * @param {array} restrictions
 *
 * @returns {object} The result set.
 */
export function RestrictAttributes(restrictions) {
  const inference = new VariantTables.Inference();
  const result = inference.restrictAttributes(restrictions);

  return result;
}

/**
 * @summary
 * The function InferenceAttributes.
 *
 * @param {string} database
 * @param {string} tableName
 * @param {object} options
 *
 * @returns {object} The result set.
 */
export function InferenceAttributes(database, tableName, options) {
  const simpleInference = new VariantTables.SimpleInference();
  const result = simpleInference.restrictAttributes(StringOrEmpty(database), StringOrEmpty(tableName), options);

  return result;
}

/**
 * @summary
 * The function BenchmarkSelectValueScript.
 *
 * @param {string} attributePath
 * @param {number} count
 *
 * @returns {number} The time in ms.
 */
export function BenchmarkSelectValueScript(attributePath, count) {
  const t0 = new Date().getTime();

  for (let i = 0; i < count; ++i) {
    const value = pcl.GetSelectedValue(attributePath);

    pcl.SelectValue(attributePath, value);
  }

  const t1 = new Date().getTime();
  const calls = 2 * count / (t1 - t0);

  return calls;
}

/**
 * @summary
 * The function RunBenchmarks.
 *
 * @param {string} attributePath
 * @param {number} count
 * @param {number} iterations
 */
export function RunBenchmarks(attributePath, count, iterations) {
  for (let i = 0; i < count; ++i) {
    const scriptCalls = BenchmarkSelectValueScript(attributePath, iterations);
    pcl.LogInformation(`Script Calls/ms: ${scriptCalls}`);

    const nativeCalls = pcl.BenchmarkSelectValueNative(attributePath, iterations);
    pcl.LogInformation(`Native Calls/ms: ${nativeCalls}`);

    pcl.LogInformation(`Script/Native Calls Ratio: ${scriptCalls/nativeCalls}`);
  }
}

/**
 * @summary
 * The function BenchmarkStringScript.
 *
 * @param {number} count
 *
 * @returns {number} The time in ms.
 */
export function BenchmarkStringScript(count) {
  const t0 = new Date().getTime();
  let sum = 0;

  for (let i = 0; i < count; ++i) {
    sum += pcl.BenchmarkStringParams("text1", "text2", "text3", "text4");
  }

  const t1 = new Date().getTime();
  const calls = 2 * count / (t1 - t0);

  return calls;
}

/**
 * @summary
 * The function BenchmarkSpanScript.
 *
 * @param {number} count
 *
 * @returns {number} The time in ms.
 */
export function BenchmarkSpanScript(count) {
  const t0 = new Date().getTime();
  let sum = 0;

  for (let i = 0; i < count; ++i) {
    sum += pcl.BenchmarkSpanParams("text1", "text2", "text3", "text4");
  }

  const t1 = new Date().getTime();
  const calls = 2 * count / (t1 - t0);

  return calls;
}

/**
 * @summary
 * The function RunStringSpanBenchmarks.
 *
 * @param {number} count
 * @param {number} iterations
 */
export function RunStringSpanBenchmarks(count, iterations) {
  for (let i = 0; i < count; ++i) {
    const nativeStringCalls = pcl.BenchmarkStringParamsNative("text", iterations);
    pcl.LogInformation(`Native String Calls/ms: ${nativeStringCalls}`);

    const stringCalls = BenchmarkStringScript(iterations);
    pcl.LogInformation(`String Calls/ms: ${stringCalls}`);

    const spanCalls = BenchmarkSpanScript(iterations);
    pcl.LogInformation(`Span Calls/ms: ${spanCalls}`);

    pcl.LogInformation(`String/Span Calls Ratio: ${stringCalls/spanCalls}`);
  }
}

/**
 * @summary
 * The function GetCurrentLanguage.
 *
 * @deprecated Use GetLocaleId instead.
 */
export function GetCurrentLanguage() {
  return pcl.GetLocaleId();
}

/**
 * @summary
 * The function GetFulfilledRules.
 *
 * @deprecated Use GetTrueConditions instead.
 */
export function GetFulfilledRules() {
  return pcl.GetTrueConditions();
}

/**
* @summary
* The function GetObjectTranslation.
*
* @deprecated Use GetObjectNameTranslation instead.
*
* @param {string} objectId The parameter objectId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetObjectTranslation(objectId, localeId) {
  return pcl.GetObjectNameTranslation(objectId, localeId);
}

/**
* @summary
* The function GetAttributeTranslation.
*
* @deprecated Use GetAttributeNameTranslation instead.
*
* @param {string} attributeId The parameter attributeId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetAttributeTranslation(attributeId, localeId) {
  return pcl.GetAttributeNameTranslation(attributeId, localeId);
}

/**
* @summary
* The function GetAttributeGroupTranslation.
*
* @deprecated Use GetAttributeGroupNameTranslation instead.
*
* @param {string} groupId The parameter groupId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetAttributeGroupTranslation(groupId, localeId) {
  return pcl.GetAttributeGroupNameTranslation(groupId, localeId);
}

/**
* @summary
* The function GetUserDefinedTranslation.
*
* @deprecated Use GetUserDefinedNameTranslation instead.
*
* @param {string} translationId The parameter translationId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetUserDefinedTranslation(translationId, localeId) {
  return pcl.GetUserDefinedNameTranslation(translationId, localeId);
}

/**
* @summary
* The function GetValueTranslation.
*
* @deprecated Use GetValueNameTranslation instead.
*
* @param {string} attributeId The parameter attributeId.
* @param {string} valueId The parameter valueId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetValueTranslation(attributeId, valueId, localeId) {
  return pcl.GetValueNameTranslation(attributeId, valueId, localeId);
}

/**
* @summary
* The function GetObjectPropertyTranslation.
*
* @deprecated Use GetObjectPropertyNameTranslation instead.
*
* @param {string} objectId The parameter objectId.
* @param {string} propertyId The parameter propertyId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetObjectPropertyTranslation(objectId, propertyId, localeId) {
  return pcl.GetObjectPropertyNameTranslation(objectId, propertyId, localeId);
}

/**
* @summary
* The function GetAttributePropertyTranslation.
*
* @deprecated Use GetAttributePropertyNameTranslation instead.
*
* @param {string} objectId The parameter objectId.
* @param {string} attributeId The parameter attributeId.
* @param {string} propertyId The parameter propertyId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetAttributePropertyTranslation(objectId, attributeId, propertyId, localeId) {
  return pcl.GetAttributePropertyNameTranslation(objectId, attributeId, propertyId, localeId);
}

/**
* @summary
* The function GetValuePropertyTranslation.
*
* @deprecated Use GetValuePropertyNameTranslation instead.
*
* @param {string} objectId The parameter objectId.
* @param {string} attributeId The parameter attributeId.
* @param {string} valueId The parameter valueId.
* @param {string} propertyId The parameter propertyId.
* @param {string} localeId The parameter localeId.
* @returns {string} The return value.
*/
export function GetValuePropertyTranslation(objectId, attributeId, valueId, propertyId, localeId) {
  return pcl.GetValuePropertyNameTranslation(objectId, attributeId, valueId, propertyId, localeId);
}

/**
* @summary
* The function GetAttributeProperty.
*
* @deprecated Use GetAttributePropertyValue instead.
*
*/
export function GetAttributeProperty(attributePath, propertyId) {
  return pcl.GetAttributePropertyValue(attributePath, propertyId);
}

/**
* @summary
* The function GetObjectProperty.
*
* @deprecated Use GetObjectPropertyValue instead.
*
*/
export function GetObjectProperty(objectPath, propertyId) {
  return pcl.GetObjectPropertyValue(objectPath, propertyId);
}

/**
* @summary
* The function GetStateProperty.
*
* @deprecated Use GetStatePropertyValue instead.
*
*/
export function GetStateProperty(propertyId) {
  return pcl.GetStatePropertyValue(propertyId);
}

/**
* @summary
* The function GetValueProperty.
*
* @deprecated Use GetValuePropertyValue instead.
*
*/
export function GetValueProperty(attributePath, valueId, propertyId) {
  return pcl.GetValuePropertyValue(attributePath, valueId, propertyId);
}

/**
* @summary
* The function SetAttributeProperty.
*
* @deprecated Use SetAttributePropertyValue instead.
*
*/
export function SetAttributeProperty(attributePath, propertyId, value) {
  return pcl.SetAttributePropertyValue(attributePath, propertyId, value);
}

/**
* @summary
* The function SetObjectProperty.
*
* @deprecated Use SetObjectPropertyValue instead.
*
*/
export function SetObjectProperty(objectPath, propertyId, value) {
  return pcl.SetObjectPropertyValue(objectPath, propertyId, value);
}

/**
* @summary
* The function SetStateProperty.
*
* @deprecated Use SetStatePropertyValue instead.
*
*/
export function SetStateProperty(propertyId, value) {
  return pcl.SetStatePropertyValue(propertyId, value);
}

/**
* @summary
* The function SetValueProperty.
*
* @deprecated Use SetValuePropertyValue instead.
*
*/
export function SetValueProperty(attributePath, valueId, propertyId, value) {
  return pcl.SetValuePropertyValue(attributePath, valueId, propertyId, value);
}
