import "./ecl";

class Store {
  _root;

  constructor() {
    this._root = {};
  }

  clear() {
    this._root = {};
  }

  saveJson() {
    return JSON.stringify(this._root);
  }

  restoreJson(json) {
    this._root = JSON.parse(json);
  }

  getPath(loc) {
    return loc
      .replaceAll("[", ".RANGE.")
      .replaceAll("]", ".")
      .split(".")
      .filter((x) => x.trim());
  }

  copy(loc1, loc2) {
    if (!store.exists(loc1)) {
      return false;
    }

    const value = store.get(loc1);

    store.set(loc2, value);

    return true;
  }

  move(loc1, loc2) {
    if (!store.exists(loc1)) {
      return false;
    }

    const value = store.get(loc1);

    store.remove(loc1);
    store.set(loc2, value);

    return true;
  }

  set(loc, value, merge = false) {
    const path = this.getPath(loc);
    let current = this._root;

    for (let i = 0; i < path.length - 1; ++i) {
      const item = current[path[i]];

      if (typeof item === "object") {
        current = current[path[i]];
      } else {
        current = current[path[i]] = {};
      }
    }

    if (merge && isobject(value)) {
      current[path[path.length - 1]] = mergeobject(
        current[path[path.length - 1]],
        this._copyOf(value)
      );
    } else {
      current[path[path.length - 1]] = this._copyOf(value);
    }
  }

  setString(loc, value) {
    const str = this._asString(value);

    return this.set(loc, str);
  }

  setInt(loc, value) {
    const num = this._asInt(value);

    return this.set(loc, num);
  }

  setFloat(loc, value) {
    const num = this._asFloat(value);

    return this.set(loc, num);
  }

  setBool(loc, value) {
    const bool = this._asBool(value);

    return this.set(loc, bool);
  }

  setObject(loc, value, merge = false) {
    const obj = this._asObject(value);

    return this.set(loc, obj, merge);
  }

  get(loc) {
    const result = this._getBase(loc);

    if (result) {
      return this._copyOf(result.value);
    }
  }

  getSize(loc) {
    const result = this._getBase(loc);

    if (result) {
      const value = result.value;

      if (!value || typeof value !== "object") {
        return 0;
      }

      if (value instanceof Array) {
        return value.length;
      }

      return Object.keys(result.value).length;
    }

    return 0;
  }

  getString(loc) {
    const value = this.get(loc);

    return this._asString(value);
  }

  getBool(loc) {
    const value = this.get(loc);

    return this._asBool(value);
  }

  getInt(loc) {
    const value = this.get(loc);

    return this._asInt(value);
  }

  getFloat(loc) {
    const value = this.get(loc);

    return this._asFloat(value);
  }

  getObject(loc) {
    const value = this.get(loc);

    return this._asObject(value);
  }

  exists(loc) {
    const result = this._getBase(loc);

    return !!result;
  }

  remove(loc) {
    const result = this._getBase(loc);

    if (result) {
      delete result.base[result.name];
    }
  }

  _getBase(loc) {
    const path = this.getPath(loc);
    let current = this._root;
    let previous = this._root;
    let previousName = "";
    let isArray = false;

    for (let i = 0; i < path.length - 1; ++i) {
      if (isArray && (path[i] === "RANGE" || path[i] === "SEQUENCE")) {
        continue;
      }

      const item = current[path[i]];

      if (!item || typeof item !== "object") {
        return null;
      }

      isArray = item instanceof Array;

      previous = current;
      previousName = path[i];
      current = current[path[i]];
    }

    if (isArray && path[path.length - 1] === "SIZE") {
      return {
        base: previous,
        name: previousName,
        value: current.length,
      };
    }

    if (
      path[path.length - 1] === "RANGE" ||
      path[path.length - 1] === "SEQUENCE"
    ) {
      return {
        base: previous,
        name: previousName,
        value: current,
      };
    }

    const value = current[path[path.length - 1]];

    return {
      base: current,
      name: path[path.length - 1],
      value,
    };
  }

  _copyOf(value) {
    if (typeof value === "object") {
      try {
        return JSON.parse(JSON.stringify(value));
      } catch (e) {
        return {};
      }
    }

    return value;
  }

  _asBool(value) {
    if (
      value === undefined ||
      value === null ||
      value === 0 ||
      value === "false" ||
      value === "FALSE" ||
      value === "" ||
      value === false
    ) {
      return 0;
    }

    return 1;
  }

  _asInt(value) {
    const parsed = parseInt(value);

    if (isNaN(parsed)) {
      return 0;
    }

    return parsed;
  }

  _asFloat(value) {
    const parsed = parseFloat(value);

    if (isNaN(parsed)) {
      return 0;
    }

    return parsed;
  }

  _asString(value) {
    const type = typeof value;

    if (type === "object" || type === "function" || type === "undefined") {
      return "";
    }

    return String(value);
  }

  _asObject(value) {
    if (typeof value === "object") {
      return value;
    }

    return {};
  }
}

const store = new Store();

function isobject(value) {
  return value && typeof value === "object" && !Array.isArray(value);
}

function mergeobject(targetObj, sourceObj) {
  if (isobject(targetObj) && isobject(sourceObj)) {
    for (const prop in sourceObj) {
      if (isobject(sourceObj[prop])) {
        if (!targetObj[prop]) {
          Object.assign(targetObj, { [prop]: {} });
        }
        mergeobject(targetObj[prop], sourceObj[prop]);
      } else {
        Object.assign(targetObj, { [prop]: sourceObj[prop] });
      }
    }
  }

  return targetObj;
}

function combinestorepath(a, b) {
  let path = a;

  if (!path) {
    path = b;
  } else if (b) {
    path += "." + b;
  }

  return path;
}

function removeprefix(path, count) {
  let pos = 0;

  for (let i = 0; i < count; ++i) {
    const findPos = path.indexOf(".", pos);

    if (findPos < 0) {
      break;
    }

    pos = findPos + 1;
  }

  return path.substring(pos);
}

function getcfgpath(path) {
  const pos = path.indexOf(".");
  const base = pos < 0 ? path : path.slice(0, pos);

  // getalias("CFG_THIS"): CONFIGURATIONS.CFG_NOTEBOOK
  // getalias("CFGDATA_THIS"): CONFIGURATIONS.CFG_NOTEBOOK.CFGDATA.CFG_NOTEBOOK
  // getalias("CONFIGURATIONS_THIS"): CONFIGURATIONS.CFG_NOTEBOOK.CONFIG.CFG_NOTEBOOK

  switch (base) {
    case "CFGDATA_THIS":
      return removeprefix(path, 1);
    case "CONFIGURATIONS_THIS":
      return removeprefix(path, 1);
    case "CFG_THIS":
      return "ROOT." + removeprefix(path, 3);
    case "CONFIGURATIONS":
      return "ROOT." + removeprefix(path, 4);
  }

  return null;
}

globalThis.move = function move(a, b) {
  const cfgPathA = getcfgpath(a);
  const cfgPathB = getcfgpath(b);

  if (cfgPathA !== null || cfgPathB !== null) {
    ECL_LogError("cfg path access not supported: " + a + ", " + b);
  } else {
    return store.move(a, b);
  }
};

globalThis.copy = function copy(a, b) {
  const cfgPathA = getcfgpath(a);
  const cfgPathB = getcfgpath(b);

  if (cfgPathA !== null || cfgPathB !== null) {
    ECL_LogError("cfg path access not supported: " + a + ", " + b);
  } else {
    return store.copy(a, b);
  }
};

globalThis.deletepath = function deletepath(a, b) {
  const path = combinestorepath(b, a);
  const cfgPath = getcfgpath(path);

  if (cfgPath !== null) {
    ECL_LogError("cfg path access not supported: " + path);
  } else {
    store.remove(path);
  }
};

globalThis.checkexist = function checkexist(a, b) {
  const path = combinestorepath(b, a);
  const cfgPath = getcfgpath(path);

  if (cfgPath !== null) {
    ECL_LogError("cfg path access not supported: " + path);

    return false;
  } else {
    return store.exists(path);
  }
};

globalThis.getsize = function getsize(a, b) {
  const path = combinestorepath(b, a);
  const cfgPath = getcfgpath(path);

  if (cfgPath !== null) {
    if (cfgPath.endsWith(".RANGE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".RANGE".length);

      if (ECL_ObjectValid(objPath))
      {
        return PCL.GetAttributeCount(ECL_ConvertLegacyPath(objPath));
      }

      if (ECL_AttributeValid(objPath))
      {
        return PCL.GetValueCount(ECL_ConvertLegacyPath(objPath));
      }
    }

    if (cfgPath.endsWith(".SEQUENCE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".SEQUENCE".length);

      if (ECL_ObjectValid(objPath))
      {
        return PCL.GetAttributeCount(ECL_ConvertLegacyPath(objPath));
      }

      if (ECL_AttributeValid(objPath))
      {
        return PCL.GetValueCount(ECL_ConvertLegacyPath(objPath));
      }
    }

    ECL_LogError("cfg path access not supported: " + path);

    return 0;
  }

  return store.getSize(path);
};

function getvalue(a, b) {
  const path = combinestorepath(b, a);
  const cfgPath = getcfgpath(path);

  if (cfgPath !== null) {
    if (path === "CONFIGURATIONS.CURRENTCONFIGID" || cfgPath === "CURRENTCONFIGID" || cfgPath === "ROOT.CURRENTCONFIGID") {
      return ECL_GetRootCfgObject();
    }

    if (cfgPath.endsWith(".CONTROL_TYPE")) {
      const objPath = cfgPath.substring(
        0,
        cfgPath.length - ".CONTROL_TYPE".length
      );

      if (ECL_IsSubConfiguration(objPath)) {
        return "";
      }

      return ECL_GetControlType(objPath);
    }

    if (cfgPath.endsWith(".OBJTYPE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".OBJTYPE".length);

      return ECL_GetObjectType(objPath);
    }

    if (cfgPath.endsWith(".CARD_MIN")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".CARD_MIN".length);

      return ECL_GetCardMin(attrPath);
    }

    if (cfgPath.endsWith(".CARD_MAX")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".CARD_MAX".length);

      return ECL_GetCardMax(attrPath);
    }

    if (cfgPath.endsWith(".VIEW_LEVEL")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".VIEW_LEVEL".length);

      return ECL_GetAttributeProperty(attrPath, "VIEW_LEVEL");
    }

    if (cfgPath.endsWith(".CFG_CONTROL")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".CFG_CONTROL".length);

      return ECL_GetAttributeProperty(attrPath, "Control");
    }

    if (cfgPath.endsWith(".REQUIRED")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".REQUIRED".length);

      return ECL_AttributeRequired(attrPath);
    }

    if (cfgPath.endsWith(".MANDATORY")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".MANDATORY".length);

      return ECL_AttributeRequired(attrPath);
    }

    if (cfgPath.endsWith(".ID")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".ID".length);

      return PCL.GetObjectId(ECL_ConvertLegacyPath(objPath)) ?? "";
    }

    if (cfgPath.endsWith(".TYPECODE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".TYPECODE".length);

      return PCL.GetObjectTypeCodeTemplate(ECL_ConvertLegacyPath(objPath)) ?? "";
    }

    if (cfgPath.endsWith(".XTYPECODE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".XTYPECODE".length);

      return PCL.GetObjectTypeCode(ECL_ConvertLegacyPath(objPath)) ?? "";
    }

    if (cfgPath.endsWith(".SEQUENCE.SIZE") || cfgPath.endsWith(".RANGE.SIZE")) {
      const objPath = cfgPath.substring(0, cfgPath.length - ".SIZE".length);

      return getsize(objPath, "");
    }

    if (cfgPath === "CLASSNAME" || cfgPath.endsWith(".CLASSNAME")) {
      return store.getString(path);
    }

    if (cfgPath.endsWith(".ADATA1")) {
      return cfgGetValue(cfgPath);
    }

    // TODO: RAW, VALUE, CDATA, ADATA, BDATA
    const value = store.get("CFG_STORE." + cfgPath);

    return value;
  }

  return store.get(path);
}

function setvalue(a, b, value) {
  const path = combinestorepath(b, a);
  const cfgPath = getcfgpath(path);

  if (cfgPath !== null) {
    if (path === "CONFIGURATIONS.CURRENTCONFIGID" || cfgPath === "CURRENTCONFIGID" || cfgPath === "ROOT.CURRENTCONFIGID") {
      // read-only
    } else if (cfgPath.endsWith(".CONTROL_TYPE")) {
      // read-only
    } else if(cfgPath.endsWith(".OBJTYPE")) {
      // read-only
    } else if (cfgPath.endsWith(".CARD_MIN")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".CARD_MIN".length);

      ECL_SetCardMin(attrPath, value);
    } else if (cfgPath.endsWith(".CARD_MAX")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".CARD_MAX".length);

      ECL_SetCardMax(attrPath, value);
    } else if (cfgPath.endsWith(".VIEW_LEVEL")) {
      // read-only
    } else if (cfgPath.endsWith(".CFG_CONTROL")) {
      // read-only
    } else if (cfgPath.endsWith(".REQUIRED")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".REQUIRED".length);

      PCL.SetAttributeRequired(ECL_ConvertLegacyPath(attrPath), store._asBool(value));
    } else if (cfgPath.endsWith(".MANDATORY")) {
      const attrPath = cfgPath.substring(0, cfgPath.length - ".MANDATORY".length);

      PCL.SetAttributeRequired(ECL_ConvertLegacyPath(attrPath), store._asBool(value));
    } else if (cfgPath.endsWith(".ID")) {
      // read-only
    } else if (cfgPath.endsWith(".TYPECODE")) {
      ECL_LogError("cfg path access not supported: " + path);
    } else if (cfgPath.endsWith(".XTYPECODE")) {
      ECL_LogError("cfg path access not supported: " + path);
    } else if (cfgPath.endsWith(".SEQUENCE.SIZE") || cfgPath.endsWith(".RANGE.SIZE")) {
      // read-only
    } else if (cfgPath.endsWith(".ADATA1")) {
      ECL_LogError("cfg path access not supported: " + path);
    } else {
      store.set("CFG_STORE." + cfgPath, value);
    }
  } else {
    store.set(path, value);
  }
};

globalThis.getstring = function getstring(a, b) {
  const value = getvalue(a, b);

  return store._asString(value);
};

globalThis.setstring = function setstring(a, b, value) {
  return setvalue(a, b, store._asString(value));
};

globalThis.getint = function getint(a, b) {
  const value = getvalue(a, b);

  return store._asInt(value);
};

globalThis.setint = function setint(a, b, value) {
  return setvalue(a, b, store._asInt(value));
};

globalThis.getdouble = function getdouble(a, b) {
  const value = getvalue(a, b);

  return store._asFloat(value);
};

globalThis.setdouble = function setdouble(a, b, value) {
  return setvalue(a, b, store._asFloat(value));
};

globalThis.getbool = function getbool(a, b) {
  const value = getvalue(a, b);

  return store._asBool(value);
};

globalThis.setbool = function setbool(a, b, value) {
  return setvalue(a, b, store._asBool(value));
};

globalThis.getobject = function getobject(a, b) {
  const value = getvalue(a, b);

  return store._asObject(value);
};

globalThis.setobject = function setobject(a, b, value) {
  return setvalue(a, b, store._asObject(value));
};

globalThis.getmapkey = function getmapkey(a, b, index) {
  const basePath = combinestorepath(b, a);
  const path = combinestorepath(basePath, index);

  ECL_LogError("getmapkey is not supported");

  return path;
};

globalThis.stringtoarray = function stringtoarray(str, path, delim) {
  const elements = StringOrEmpty(str).split(delim);

  for (let i = 0; i < elements.length; ++i) {
    const indexPath = combinestorepath(path, `[${i}]`);
    const valuePath = combinestorepath(indexPath, "VALUE");
    const value = elements[i];

    setstring(valuePath, "", value);
  }

  return elements.length;
};

globalThis.getalias = function getalias(alias) {
  switch (alias) {
    case "CFG_THIS":
      return "CONFIGURATIONS." + ECL_GetRootCfgObject();
    case "CONFIGURATIONS_THIS":
      return (
        "CONFIGURATIONS." +
        ECL_GetRootCfgObject() +
        ".CONFIG." +
        ECL_GetCurrentCfgStructurePath()
      );
    case "CFGDATA_THIS":
      return (
        "CONFIGURATIONS." +
        ECL_GetRootCfgObject() +
        ".CFGDATA." +
        ECL_GetCurrentCfgStructurePath()
      );
  }

  ECL_LogError("getalias is not supported: " + alias);

  return alias;
};

Object.fromStorage = function (path) {
  ECL_LogWarning("Object.fromStorage support is experimental: " + path);

  const obj = store.getObject(path);

  return obj;
};

Object.defineProperty(Object.prototype, "toStorage", {
  enumerable: false,
  value: function (path, merge = true) {
    ECL_LogWarning("Object.toStorage support is experimental: " + path);

    store.setObject(path, this, merge);
  },
});

globalThis.ECL_LegacyStore = {
  get RootObject() {
    return store._root;
  },
};

globalThis.OnSaveStateInternal = function () {
  ECL_SetStoreValue("LegacyStore", store.saveJson());
};

globalThis.OnRestoreStateInternal = function () {
  const json = ECL_GetStoreValue("LegacyStore");

  try {
    if (json) {
      store.restoreJson(json);
    }
  } catch (e) {
    ECL_LogError("failed to restore store: " + e.message);
  }
};
