import * as ecl from "../eclbase";
import "./Logger";
import "./Util";

if (typeof VariantTables === "undefined") {
  globalThis.VariantTables = {};
}

VariantTables.Inference = function (name) {
  this._name = name;
  this._logger = VariantTables.Logger.getLogger("Inference", name);
};

VariantTables.Inference.RestrictAll = "all";
VariantTables.Inference.RestrictNone = "none";
VariantTables.Inference.RestrictUnion = "union";
VariantTables.Inference.RestrictIntersect = "intersect";

VariantTables.Inference.prototype.restrictAttributes = function (restrictions) {
  var deltaTime = VariantTables.Util.getTime();

  for (var i = 0; i < restrictions.length; ++i) {
    var restriction = restrictions[i];

    this.restrictAttribute(restriction.attribute, restriction.values, restriction.options);
  }

  deltaTime = VariantTables.Util.getTime() - deltaTime;

  this._logger.trace("restrict attributes time:", deltaTime, "ms");
};

VariantTables.Inference.prototype.restrictAttribute = function (attribute, values, options) {
  var deltaTime = VariantTables.Util.getTime();

  options = options || {};

  var type = options.type;
  var inverse = options.inverse;
  var hasWildcards = this._hasWildcards(values, options.wildcards);
  var noAutoDeselect = options.noAutoDeselect;
  var autoSelect = options.autoSelect;

  if (!type) {
    type = VariantTables.Inference.RestrictAll;
  }

  if (type === VariantTables.Inference.RestrictUnion) {
    this._restrictAttributeUnion(attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect);
  }
  else if (type === VariantTables.Inference.RestrictIntersect) {
    this._restrictAttributeIntersect(attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect);
  }
  else if (type === VariantTables.Inference.RestrictAll) {
    this._restrictAttributeAll(attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect);
  }
  else if (type === VariantTables.Inference.RestrictNone) {
    this._restrictAttributeNone(attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect);
  }
  else {
    this._logger.error("using restrict all, unknown restriction type:", type);
    this._restrictAttributeAll(attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect);
  }

  deltaTime = VariantTables.Util.getTime() - deltaTime;

  this._logger.trace("restrict attribute time:", deltaTime, "ms");
};

VariantTables.Inference.prototype._hasWildcards = function (values, wildcards) {
  if (typeof wildcards === "object" && wildcards.length > 0) {
    for (var i = 0; i < values.length; ++i) {
      var value = values[i];

      if (VariantTables.Util.indexOf(wildcards, value) >= 0) {
        return true;
      }
    }
  }

  return false;
};

VariantTables.Inference.prototype._restrictAttributeAll = function (attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect) {
  var selections = this.getSelections(attribute);
  var map = this._getValuesMap(values);
  var allValues = this.getAllValues(attribute);

  for (var i = 0; i < allValues.length; ++i) {
    var value = allValues[i];

    if (("" + value) in map) {
      if (inverse) {
        ecl.ECL_DeactivateValue(attribute, value);
      }
      else {
        ecl.ECL_ActivateValue(attribute, value);
      }
    }
    else {
      if (inverse) {
        ecl.ECL_ActivateValue(attribute, value);
      }
      else {
        ecl.ECL_DeactivateValue(attribute, value);
      }
    }
  }

  if (!noAutoDeselect) {
    this._restrictSelections(attribute, selections, map, inverse, hasWildcards);
  }

  this._handleAutoSelect(attribute, values, inverse, hasWildcards, autoSelect);
};

VariantTables.Inference.prototype._restrictAttributeNone = function (attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect) {
  // noop
};

VariantTables.Inference.prototype._restrictAttributeUnion = function (attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect) {
  var selections = this.getSelections(attribute);
  var map = this._getValuesMap(values);

  if (inverse) {
    var allValues = this.getAllValues(attribute);

    for (var i = 0; i < allValues.length; ++i) {
      var value = allValues[i];

      if (!(("" + value) in map)) {
        ecl.ECL_ActivateValue(attribute, value);
      }
    }
  }
  else {
    for (var i = 0; i < values.length; ++i) {
      var value = values[i];

      if (!ecl.ECL_ValueActive(attribute, value)) {
        ecl.ECL_ActivateValue(attribute, value);
      }
    }
  }

  if (!noAutoDeselect) {
    this._restrictSelections(attribute, selections, map, inverse, hasWildcards);
  }

  this._handleAutoSelect(attribute, values, inverse, hasWildcards, autoSelect);
};

VariantTables.Inference.prototype._restrictAttributeIntersect = function (attribute, values, inverse, noAutoDeselect, hasWildcards, autoSelect) {
  var selections = this.getSelections(attribute);
  var map = this._getValuesMap(values);

  var activeValues = ecl.ECL_GetAllActiveAttributeValues(attribute);
  for (var i = 0; i < activeValues.length; ++i) {
    var value = activeValues[i];

    if (inverse) {
      if (("" + value) in map) {
        ecl.ECL_DeactivateValue(attribute, value);
      }
    }
    else {
      if (!(("" + value) in map)) {
        ecl.ECL_DeactivateValue(attribute, value);
      }
    }
  }

  if (!noAutoDeselect) {
    this._restrictSelections(attribute, selections, map, inverse, hasWildcards);
  }

  this._handleAutoSelect(attribute, values, inverse, hasWildcards, autoSelect);
};

VariantTables.Inference.prototype._getValuesMap = function (values) {
  var map = {};

  if (values === null) {
    return map;
  }

  for (var i = 0; i < values.length; ++i) {
    var value = "" + values[i];

    map[value] = true;
  }

  return map;
};

VariantTables.Inference.prototype.getSelections = function (attribute) {
  var selections;

  if (ecl.ECL_AttributeEditable(attribute)) {
    var value = ecl.ECL_GetValue(attribute);

    if (value === true || value === false) {
      value = value ? 1 : 0;
    }

    if (value === "" || typeof value === "undefined") {
      selections = [];
    } else {
      selections = [value];
    }
  }
  else {
    selections = ecl
      .ECL_GetAllSelectedAttributeValues(attribute)
      .map(x => (x === true || x === false) ? (x ? 1 : 0) : x);
  }

  return selections;
};

VariantTables.Inference.prototype.getAllValues = function (attribute) {
  var allValues = ecl
    .ECL_GetAllAttributeValues(attribute)
    .map(x => (x === true || x === false) ? (x ? 1 : 0) : x);

  return allValues;
};

VariantTables.Inference.prototype._restrictSelections = function (attribute, selections, valueMap, inverse, hasWildcards) {
  if (selections.length === 0) {
    return;
  }

  if (hasWildcards) {
    if (inverse) {
      ecl.ECL_ResetAttribute(attribute);
    }
  } else {
    if (ecl.ECL_AttributeEditable(attribute)) {
      // TODO: currently no deselect for EDIT --> behavior not clear, e.g. for operator >, <, ...
      /*for (var i=0; i<selections.length; ++i) {
        var value = selections[i];
        var hasValue = (("" + value) in valueMap);

        if ((inverse && hasValue) || (!inverse && !hasValue)) {
          ecl.ECL_ResetAttribute(attribute);
        }
      }*/
    } else {
      for (var i = 0; i < selections.length; ++i) {
        var value = selections[i];

        if (!ecl.ECL_ValueActive(attribute, value)) {
          ecl.ECL_DeselectValue(attribute, value);
        }
      }
    }
  }
};

VariantTables.Inference.prototype._handleAutoSelect = function (attribute, values, inverse, hasWildcards, autoSelect) {
  var selections = this.getSelections(attribute);

  if (selections.length > 0) {
    return;
  }

  var hasAutoSelect = autoSelect || ecl.ECL_AttributeAutoSelected(attribute);
  if (hasAutoSelect) {
    if (ecl.ECL_AttributeEditable(attribute)) {
      if (!inverse && !hasWildcards && values.length === 1) {
        ecl.ECL_SetValue(attribute, values[0]);
      }
    } else {
      var activeValues = ecl.ECL_GetAllActiveAttributeValues(attribute);

      if (activeValues.length === 1) {
        ecl.ECL_SelectValue(attribute, activeValues[0]);
      }
    }
  }
};
