import * as ecl from "../eclbase";
import * as pcl from "../pcl";
import "./Util";
import "./Logger";
import "./Cache";
import "./QueryBuilder";

if (typeof VariantTables === "undefined") {
  globalThis.VariantTables = {};
}

VariantTables.Query = function (name) {
  this._name = name;
  this._cumulativeQueryTime = 0;
  this._logger = VariantTables.Logger.getLogger("Query", name);
};

VariantTables.Query.prototype.queryStages = function (stages) {
  for (var i = 0; i < stages.length; ++i) {
    this.queryStage(stages[i]);
  }
};

VariantTables.Query.prototype.queryStage = function (stage) {
  if (stage.disabled) {
    return;
  }

  for (var i = 0; i < stage.queries.length; ++i) {
    this.query(stage.queries[i]);
  }
};

VariantTables.Query.prototype.query = function (query) {
  if (query.disabled) {
    return;
  }

  var builder = new VariantTables.QueryBuilder();
  var nativeQueries = builder.buildQueries(query);
  var cache = VariantTables.Cache.instance();

  for (var i = 0; i < nativeQueries.length; ++i) {
    var nativeQuery = nativeQueries[i];

    this._nativeQuery(query, nativeQuery, i);
  }

  this._logger.trace("cache hit rate:", cache.getHitRate());
};

VariantTables.Query.prototype._makeKey = function (nativeQuery) {
  var fullQuery = [];
  var parts = nativeQuery.query.split("?");
  var params = nativeQuery.params;

  for (var i = 0; i < parts.length; ++i) {
    fullQuery.push(parts[i]);

    if (i < params.length) {
      var value = params[i];

      if (typeof value === "number") {
        fullQuery.push(params[i]);
      } else if (typeof value === "undefined" || value === null) {
        fullQuery.push("null");
      } else if (typeof value === "boolean") {
        fullQuery.push(value ? 1 : 0);
      } else {
        fullQuery.push("'" + value.replaceAll("'", "''") + "'");
      }
    }
  }

  var key = "DB: " + nativeQuery.database + ", Query: " + fullQuery.join("");

  return key;
};

VariantTables.Query.prototype._nativeQuery = function (query, nativeQuery, index) {
  var cache = VariantTables.Cache.instance();
  var key = this._makeKey(nativeQuery);
  var results = cache.get(key);
  var status = {
    error: false,
    message: ""
  };

  if (results === null) {
    var deltaTime = VariantTables.Util.getTime();
    var rows = ecl.ECL_ObjectFromDatabase(nativeQuery.database, nativeQuery.query, nativeQuery.params);
    var errorString = ecl.ECL_GetLastDatabaseError(nativeQuery.database, true);

    if (errorString) {
      status.error = true;
      status.message = errorString;
      this._logger.error("query error:", errorString, ",", key);
    }

    if (nativeQuery.selectGroups) {
      results = this._processSelectGroupsResult(nativeQuery.columns, rows);
    }
    else {
      results = this._processSelectResult(nativeQuery.columns, rows);
    }

    cache.set(key, results);

    deltaTime = VariantTables.Util.getTime() - deltaTime;

    this._cumulativeQueryTime += deltaTime;

    this._logger.trace("query time:", deltaTime, "ms, for", key);
    this._logger.trace("cumulative query time:", this._cumulativeQueryTime, "ms");
  } else {
    this._logger.trace("query time:", 0, "ms (cache hit), for", key);
    this._logger.trace("cumulative query time:", this._cumulativeQueryTime, "ms");
  }

  for (var i = 0; i < results.length; ++i) {
    var result = results[i];

    query.callback(status, result, index + i);
  }
};

VariantTables.Query.prototype._processSelectResult = function (columns, rows) {
  var values = new Array(rows.length);

  for (var i = 0; i < rows.length; ++i) {
    var row = rows[i];

    if (columns.length === 1) {
      var canonicalColumn = pcl.DatabaseUpperCaseColumns() ? columns[0].toUpperCase() : columns[0];
      var value = row[canonicalColumn];

      values[i] = value;
    }
    else {
      values[i] = [];

      for (var j = 0; j < columns.length; ++j) {
        var canonicalColumn = pcl.DatabaseUpperCaseColumns() ? columns[j].toUpperCase() : columns[j];
        var value = row[canonicalColumn];

        values[i].push(value);
      }
    }
  }

  var results = [{
    column: columns[0],
    columns: columns,
    values: values
  }];

  return results;
};

VariantTables.Query.prototype._processSelectGroupsResult = function (columns, rows) {
  var results = [];
  var canonicalKeyColumn = pcl.DatabaseUpperCaseColumns() ? columns[0].toUpperCase() : columns[0];
  var canonicalValueColumn = pcl.DatabaseUpperCaseColumns() ? columns[1].toUpperCase() : columns[1];
  var lastKey = null;

  for (var i = 0; i < rows.length; ++i) {
    var row = rows[i];
    var key = row[canonicalKeyColumn];
    var value = row[canonicalValueColumn];

    if (key !== lastKey) {
      var result = {
        column: key,
        columns: [key],
        values: []
      };

      results.push(result);
      lastKey = key;
    }

    results[results.length - 1].values.push(value);
  }

  return results;
};
