import "./Logger";

if (typeof VariantTables === "undefined") {
  globalThis.VariantTables = {};
}

VariantTables.QueryBuilder = function (name) {
  this._name = name;
  this._logger = VariantTables.Logger.getLogger("QueryBuilder", name);
};

VariantTables.QueryBuilder.prototype.buildQueries = function (query) {
  this._quoter = VariantTables.DatabaseInfo.getQuoteFunction(query.database);
  this._dbType = VariantTables.DatabaseInfo.GetDatabaseProvider(query.database);

  if (query.selectGroups) {
    var results = this._buildSelectGroupsQueries(query);

    return results;
  }

  var results = this._buildSelectQueries(query);

  return results;
};

VariantTables.QueryBuilder.prototype._buildSelectQueries = function (query) {
  var results = [];

  for (var i = 0; i < query.select.length; ++i) {
    var column = query.select[i];
    var columns = (typeof column === "object") ? column : [column];
    var selectFromString = this._buildSelectFrom(query, columns);
    var conditionsObj = this._buildWhere(query, columns[0]);
    var queryList = [selectFromString, conditionsObj.where];
    var queryString = queryList.join(" ");

    var result = {
      selectGroups: false,
      database: query.database,
      from: query.from,
      columns: columns,
      query: queryString,
      params: conditionsObj.params
    };

    results.push(result);
  }

  return results;
};

VariantTables.QueryBuilder.prototype._buildSelectGroupsQueries = function (query) {
  var results = [];

  var columns = [query.selectGroups.groupBy, query.selectGroups.values];
  var selectFromString = this._buildSelectFrom(query, columns);
  var conditionsObj = this._buildWhere(query, "");
  var orderString = this._orderBy(query.selectGroups.groupBy);
  var queryList = [selectFromString, conditionsObj.where, orderString];
  var queryString = queryList.join(" ");

  var result = {
    selectGroups: true,
    database: query.database,
    from: query.from,
    columns: columns,
    query: queryString,
    params: conditionsObj.params
  };

  results.push(result);

  return results;
};

VariantTables.QueryBuilder.prototype._buildSelectFrom = function (query, columns) {
  var selectFromList = [];

  selectFromList.push("SELECT DISTINCT");

  for (var i = 0; i < columns.length; ++i) {
    var column = columns[i];

    if (i > 0) {
      selectFromList.push(",");
    }

    selectFromList.push(this._quoter(column));
  }

  selectFromList.push("FROM");
  selectFromList.push(this._quoter(query.from));

  var result = selectFromList.join(" ");

  return result;
};

VariantTables.QueryBuilder.prototype._orderBy = function (column) {
  var orderList = [];

  orderList.push("ORDER BY");
  orderList.push(this._quoter(column));
  orderList.push("ASC");

  var orderString = orderList.join(" ");

  return orderString;
};

VariantTables.QueryBuilder.prototype._buildWhere = function (query, column) {
  var whereString = "";

  if (!query.where) {
    return whereString;
  }

  var whereList = [];
  var paramList = [];

  whereList.push("WHERE");

  this._buildGroup(query, query.where, whereList, paramList, column);

  if (whereList.length > 1) {
    whereString = whereList.join(" ");
  }

  var result = {
    where: whereString,
    params: paramList
  };

  return result;
};

VariantTables.QueryBuilder.prototype._buildGroup = function (query, group, whereList, paramList, column) {
  if (!group.conditions) {
    this._buildCondition(query, group, whereList, paramList, column);
    return;
  }

  var operator;

  if (group.operator === "and") {
    operator = "AND";
  }
  else if (group.operator === "or") {
    operator = "OR";
  }
  else {
    this._logger.error("using AND, unknown operator:", operator);
    operator = "AND";
  }

  if (group.not) {
    operator = "NOT " + operator;
  }

  whereList.push("(");

  var first = true;
  for (var i = 0; i < group.conditions.length; ++i) {
    var condition = group.conditions[i];

    if (!first) {
      whereList.push(operator);
    }

    var whereSize = whereList.length;

    this._buildGroup(query, condition, whereList, paramList, column);

    if (whereSize < whereList.length) {
      if (first) {
        first = false;
      }
    }
    else if (!first) {
      whereList.pop();
    }
  }

  whereList.push(")");

  if (first) {
    whereList.pop();
    whereList.pop();
  }
};

VariantTables.QueryBuilder.prototype._buildCondition = function (query, condition, whereList, paramList, column) {
  if (condition.disabled || (column === condition.column && !condition.noAutoDeactivate)) {
    return;
  }

  var isEmpty = false;
  var not = false;
  var operator = condition.operator;
  var whereLength = whereList.length;

  whereList.push("(");

  switch (operator) {
    case "custom":
      if (condition.expression !== "") {
        whereList.push("(");
        whereList.push(condition.expression);
        whereList.push(")");
        for (var i = 0; condition.params && i < condition.params.length; ++i) {
          var param = condition.params[i];

          paramList.push(param);
        }
      }
      break;
    case "eq":
      whereList.push(this._quoter(condition.column));
      if (condition.values[0] !== null) {
        whereList.push(condition.not ? "<>" : "=");
        whereList.push("?");
        paramList.push(condition.values[0]);
      }
      else {
        whereList.push(condition.not ? "IS NOT NULL" : "IS NULL");
      }
      break;
    case "like":
      whereList.push(this._quoter(condition.column));
      whereList.push(condition.not ? "NOT LIKE" : "LIKE");
      whereList.push("?");
      paramList.push("%" + condition.values[0] + "%");
      break;
    case "contains":
      whereList.push(condition.not ? "NOT " : "");
      if (this._dbType === "Npgsql") {
        whereList.push("(strpos(");
        whereList.push(this._quoter(condition.column));
        whereList.push(", ?) > 0)");
      } else if (this._dbType === "Microsoft.Data.SqlClient") {
        whereList.push("(CHARINDEX(?, ");
        whereList.push(this._quoter(condition.column));
        whereList.push(") > 0)");
      } else {
        whereList.push("(INSTR(");
        whereList.push(this._quoter(condition.column));
        whereList.push(", ?) > 0)");
      }
      paramList.push(condition.values[0]);
      break;
    case "containsElement":
      if (this._dbType === "Npgsql") {
        whereList.push(condition.not ? "NOT " : "");
        whereList.push("(strpos(");
        whereList.push(this._quoter(condition.column));
        whereList.push(", ?) > 0 OR ");
        whereList.push(this._quoter(condition.column));
        whereList.push(" = ?)");
      } else if (this._dbType === "Microsoft.Data.SqlClient") {
        whereList.push(condition.not ? "NOT " : "");
        whereList.push("(CHARINDEX(?, ");
        whereList.push(this._quoter(condition.column));
        whereList.push(") > 0 OR ");
        whereList.push(this._quoter(condition.column));
        whereList.push(" = ?)");
      } else {
        whereList.push(condition.not ? "NOT " : "");
        whereList.push("(INSTR(");
        whereList.push(this._quoter(condition.column));
        whereList.push(", ?) > 0 OR ");
        whereList.push(this._quoter(condition.column));
        whereList.push(" = ?)");
      }
      paramList.push(condition.values[1] + condition.values[0] + condition.values[1]);
      paramList.push(condition.values[0]);
      break;
    case "neq":
      whereList.push(this._quoter(condition.column));
      if (condition.values[0] !== null) {
        whereList.push(condition.not ? "=" : "<>");
        whereList.push("?");
        paramList.push(condition.values[0]);
      }
      else {
        whereList.push(condition.not ? "IS NULL" : "IS NOT NULL");
      }
      break;
    case "gt":
      whereList.push(this._quoter(condition.column));
      whereList.push(condition.not ? "<=" : ">");
      whereList.push("?");
      paramList.push(condition.values[0]);
      break;
    case "gte":
      whereList.push(this._quoter(condition.column));
      whereList.push(condition.not ? "<" : ">=");
      whereList.push("?");
      paramList.push(condition.values[0]);
      break;
    case "lt":
      whereList.push(this._quoter(condition.column));
      whereList.push(condition.not ? ">=" : "<");
      whereList.push("?");
      paramList.push(condition.values[0]);
      break;
    case "lte":
      whereList.push(this._quoter(condition.column));
      whereList.push(condition.not ? ">" : "<=");
      whereList.push("?");
      paramList.push(condition.values[0]);
      break;
    case "between":
      whereList.push(this._quoter(condition.column));
      if (condition.not) {
        whereList.push("NOT");
      }
      whereList.push("BETWEEN");
      whereList.push("?");
      whereList.push("AND");
      whereList.push("?");
      paramList.push(condition.values[0]);
      paramList.push(condition.values[1]);
      break;
    case "not in":
      not = true;
    case "in":
      this._buildInCondition(query, condition, whereList, paramList, column, not);
      break;
    case "not exists":
      not = true;
    case "exists":
      this._buildExistsCondition(query, condition, whereList, paramList, column, not);
      break;
    default:
      this._logger.error("ignoring condition, unknown operator:", operator);
      isEmpty = true;
  }

  whereList.push(")");

  if (isEmpty || whereLength + 2 === whereList.length) {
    whereList.pop();
    whereList.pop();
  }
};

VariantTables.QueryBuilder.prototype._buildExistsCondition = function (query, condition, whereList, paramList, column, not) {
  if (condition.disabled || (column === condition.column && !condition.noAutoDeactivate)) {
    return;
  }

  var newQuery = {
    database: query.database,
    from: query.from,
    select: [column],
    where: condition.condition
  };

  var selectFromString = this._buildSelectFrom(newQuery, [column]);
  var conditionsObj = this._buildWhere(newQuery, column);

  whereList.push("EXISTS");
  whereList.push("(");
  whereList.push(selectFromString);
  whereList.push(conditionsObj.where);
  whereList.push(")");

  for (var i = 0; i < conditionsObj.params.length; ++i) {
    var param = conditionsObj.params[i];

    paramList.push(param);
  }
};

VariantTables.QueryBuilder.prototype._buildInCondition = function (query, condition, whereList, paramList, column, not) {
  if (condition.disabled || (column === condition.column && !condition.noAutoDeactivate)) {
    return;
  }

  var hasNullValue = false;
  var values = [];

  for (var i = 0; i < condition.values.length; ++i) {
    var value = condition.values[i];

    if (value === null) {
      hasNullValue = true;
    }
    else {
      values.push(value);
    }
  }

  if (hasNullValue && values.length > 0) {
    whereList.push("(");
  }

  if (values.length > 0) {
    whereList.push("(");
    whereList.push(this._quoter(condition.column));
    if (condition.not && !not || !condition.not && not) {
      whereList.push("NOT");
    }
    whereList.push("IN");
    whereList.push("(");
    for (var i = 0; i < values.length; ++i) {
      if (i > 0) {
        whereList.push(",");
      }

      whereList.push("?");
      paramList.push(values[i]);
    }
    whereList.push(")");
    whereList.push(")");
  }

  if (hasNullValue) {
    if (values.length > 0) {
      whereList.push(condition.not ? "AND" : "OR");
    }

    whereList.push("(");
    whereList.push(this._quoter(condition.column));
    whereList.push(condition.not ? "IS NOT NULL" : "IS NULL");
    whereList.push(")");

    if (values.length > 0) {
      whereList.push(")");
    }
  }
};
