import * as ecl from "../eclbase";
import "./Util";
import "./Logger";
import "./DatabaseInfo";

if (typeof VariantTables === "undefined") {
  globalThis.VariantTables = {};
}

/**
 * @namespace VariantTables
 */

/**
 * @typedef {Object} VariantTables.SimpleInferenceOptions Inference options
 * @property {string} [inferenceType="all"] The inference type to use when restricting the attributes (all, union, intersect, none).
 * @property {boolean} [inferenceInverse=false] Invert the result of the variant table query (for negative variant tables).
 * @property {string[]} [wildcards=[]] Defines values that will not be restricted (e.g. NULL or *).
 * @property {string} [separator=undefined] Defines how to split non atomic values (e.g. ";" or "|").
 * @property {boolean} [noAutoDeselect=false] Do not deselect invalid values.
 * @property {boolean} [autoSelect=false] Auto select if only one value is remaining.
 * @property {Object.<string, string>} [mapping={}] Optional mapping of attribute names to database columns. A mapping to null means ignoring the attribute.
 * @property {Object.<string, (string|VariantTables.QueryOperator)>} [operators={}] Override operators per attribute. Operator: eq, neq, lt, lte, gt, gte, in, not in, like, between and exists. Default operator is eq.
 * @property {VariantTables.QueryCondition[]} [conditions=[]] Additional conditions for the database query, multiple conditions will be AND connected.
 */

/**
 * @typedef {Object} VariantTables.QueryOperator Query condition
 * @property {string} [operator] Override the operator to use: eq, neq, lt, lte, gt, gte, in, not in, like, between, contains, containsElement and exists.
 * @property {string} [inferenceType] Override the inference type.
 * @property {boolean} [inferenceInverse] Override inverse inference.
 * @property {Array} [wildcards] Override the wildcards to use.
 * @property {string} [separator=undefined] Override how to split non atomic values (e.g. ";" or "|").
 * @property {boolean} [noAutoDeselect=false] Override if invalid values should be deselected.
 * @property {boolean} [autoSelect=false] Auto select if only one value is remaining.
 * @property {(VariantTables.QueryCondition|VariantTables.QueryConditionSet|VariantTables.QueryConditionExist)} [condition] Override the condition for this attribute.
 */

/**
 * @typedef {Object} VariantTables.QueryCondition Query condition
 * @property {string} operator The condition operator: eq, neq, lt, lte, gt, gte, in, not in, like, between, contains, containsElement and exists.
 * @property {string} column The database column to apply the condition to.
 * @property {Array} values The values to pass to the operator.
 */

/**
 * @typedef {Object} VariantTables.QueryConditionSet Query condition set
 * @property {string} operator The condition operator: and and or.
 * @property {string} column The database column to apply the condition to.
 * @property {Array.<(VariantTables.QueryCondition|VariantTables.QueryConditionSet|VariantTables.QueryConditionExist)>} conditions The the sub-conditions for the operator.
 */

/**
 * @typedef {Object} VariantTables.QueryConditionExist Query exists condition
 * @property {string} operator The condition operator: and and or.
 * @property {string} column The database column to apply the condition to.
 * @property {(VariantTables.QueryCondition|VariantTables.QueryConditionSet)} condition The the sub-conditions for the operator.
 */

/**
 * Provides an easy to use interface to variant tables.
 *
 * @param {string} [name] A user-defined name for this instance, the name is shown in the log.
 * @constructor
 */
VariantTables.SimpleInference = function (name) {
  this._name = name;
  this._logger = VariantTables.Logger.getLogger("SimpleInference", name);
};

/**
 * Restrict attribute values using the given database and table.
 *
 * @param {string} database The database alias where the variant table is defined, must not be empty.
 * @param {string} tableName The name of the database table without quotes, must not be empty.
 * @param {VariantTables.SimpleInferenceOptions} [options] The restriction options.
 *
 * @return {Object} result The result of the operation.
 * @return {boolean} result.error True successful, false error.
 *
 * @example
  // DB column names match attribute names
  simpleInference.restrictAttributes(db, "VT1");

  // map attribute names to DB columns
  simpleInference.restrictAttributes(db, "VT1", {
    mapping: {
      "A_VOLTAGE": "Voltage",
      "A_POWER": "Power",
      "A_TORQUE": "Torque",
      "A_PROTECTION": "Protection",
      "A_MOTOR": "Motor",
      "A_GEARBOX": "Gearbox",
      "A_COLOR": "Color"
    }
  });

  // local overrides 1
  simpleInference.restrictAttributes(db, "VT1", {
    mapping: {
      "A_VOLTAGE": "Voltage",
      "A_POWER": "Power",
      "A_TORQUE": "Torque",
      "A_PROTECTION": "Protection",
      "A_MOTOR": "Motor",
      "A_GEARBOX": "Gearbox",
      "A_COLOR": "Color"
    },
    operators: {
      "A_POWER": "gte"
    }
  });

  // local overrides
  simpleInference.restrictAttributes(db, "VT1", {
    operators: {
      "A_POWER": "gte"
    }
  });

  // local overrides
  simpleInference.restrictAttributes(db, "VT1", {
    operators: {
      "A_POWER": {
        inferenceType: VariantTables.Inference.RestrictAll,
        inferenceInverse: true,
        condition: {
          operator: "in",
          column: "Power",
          values: [1, 2, 3]
        }
      }
    }
  });

  // local overrides
  simpleInference.restrictAttributes(db, "VT1", {
    operators: {
      "A_POWER": {
        condition: {
          operator: "between",
          column: "Power",
          values: [1, 2]
        }
      }
    }
  });

  // local overrides
  simpleInference.restrictAttributes(db, "VT1", {
    operators: {
      "A_POWER": {
        inferenceType: VariantTables.Inference.RestrictAll,
        inferenceInverse: true,
        operator: "neq",
        wildcards: [null, "-"]
      }
    }
  });

  // local overrides
  simpleInference.restrictAttributes(db, "VT1", {
    operators: {
      "A_POWER": {
        inferenceType: VariantTables.Inference.RestrictAll,
        inferenceInverse: true,
        operator: "neq",
        wildcards: [null, "-"],
        separator: "|"
      }
    }
  });

  // ignore attribute
  simpleInference.restrictAttributes(db, "VT1", {
    mapping: {
      "A_ID": null
    }
  });

  // inference type and inverse, wildcard values
  simpleInference.restrictAttributes(db, "VT1", {
    inferenceType: VariantTables.Inference.RestrictAll,
    inferenceInverse: false,
    wildcards: [null, "*"],
    separator: ";",
    noAutoDeselect: true
  });

  // additional conditions
  simpleInference.restrictAttributes(db, "VT1", {
    conditions: [
      {
        operator: "eq",
        column: "Test1",
        values: ["test"]
      }
    ]
  });

  // additional conditions complex
  simpleInference.restrictAttributes(db, "VT1", {
    conditions: [
      {
        operator: "or",
        conditions: [
          {
            operator: "eq",
            column: "A_SCREEN_SIZE",
            values: ["*"]
          },
          {
            operator: "containsElement",
            column: "A_SCREEN_SIZE",
            values: ["test", ";"]
          }
        ]
      }
    ]
  });

  // virtual conditions attribute
  simpleInference.restrictAttributes(db, "VT1", {
    wildcards: ["*"],
    separator: ";",
    conditions: [
      {
        operator: "containsElement",
        attribute: "A_SCREEN_SIZE",
        values: ["test"]
      }
    ]
  });
 */
VariantTables.SimpleInference.prototype.restrictAttributes = function (database, tableName, options) {
  var deltaTime = VariantTables.Util.getTime();

  var restrictOptions = options || {};
  var operators = restrictOptions.operators || {};
  var attrMap = restrictOptions.mapping || {};
  var wildcards = restrictOptions.wildcards || [];
  var separator = restrictOptions.separator;
  var addConditions = restrictOptions.conditions || [];
  var noAutoDeselect = !!restrictOptions.noAutoDeselect;
  var autoSelect = !!restrictOptions.autoSelect;
  var inferenceType = restrictOptions.inferenceType || VariantTables.Inference.RestrictAll;
  var inferenceInverse = !!restrictOptions.inferenceInverse;

  var dbAttributes = this._getDatabaseAttributes(database, tableName, attrMap);
  var dbColumns = this._getDatabaseColumns(database, tableName);

  if (dbAttributes.length < 1) {
    ecl.ECL_LogError(`restrictAttributes: no columns and attributes matched in table ${tableName} of database ${database}`);
  }

  var inference = new VariantTables.Inference(this._name);
  var query = new VariantTables.Query(this._name);
  var conditions = [];
  var inferenceMap = {};
  var me = this;

  var result = {
    error: false,
    data: {}
  };

  for (var i = 0; i < dbAttributes.length; ++i) {
    var attribute = dbAttributes[i].attribute;
    var column = dbAttributes[i].column;
    var columnDataType = dbAttributes[i].dataType;
    var operator = operators[attribute] || "eq";

    var localOperator;
    var localWildcards;
    var localSeparator;
    var localInferenceType;
    var localInferenceInverse;
    var localNoAutoDeselect;
    var localAutoSelect;
    var localValues;

    if (typeof operator === "string") {
      localOperator = operator;
      localWildcards = wildcards;
      localSeparator = separator;
      localInferenceType = inferenceType;
      localInferenceInverse = inferenceInverse;
      localNoAutoDeselect = noAutoDeselect;
      localAutoSelect = autoSelect;
      localValues = null;
    } else {
      localOperator = operator.operator || "eq";
      localWildcards = operator.wildcards || wildcards;
      localSeparator = typeof operator.separator !== "undefined" ? operator.separator : separator;
      localInferenceType = operator.inferenceType || inferenceType;
      localInferenceInverse = typeof operator.inferenceInverse !== "undefined" ? operator.inferenceInverse : inferenceInverse;
      localNoAutoDeselect = typeof operator.noAutoDeselect !== "undefined" ? operator.noAutoDeselect : noAutoDeselect;
      localAutoSelect = typeof operator.autoSelect !== "undefined" ? operator.autoSelect : autoSelect;
      localValues = operator.values;
    }

    // if (localWildcards
    //     && typeof localWildcards === "object"
    //     && localWildcards.length > 0
    //     && (columnDataType === "BOOL" || columnDataType === "INTEGER" || columnDataType === "DOUBLE")) {
    //   localWildcards = [null];
    // }

    inferenceMap[attribute] = {
      type: localInferenceType,
      wildcards: localWildcards,
      separator: localSeparator,
      inverse: localInferenceInverse,
      noAutoDeselect: localNoAutoDeselect,
      autoSelect: localAutoSelect
    };

    var values = localValues || inference.getSelections(attribute);
    var condition = {
      disabled: values.length === 0 || (values.length === 1 && values[0] === ""),
      operator: "or",
      conditions: []
    };

    if (operator.condition) {
      condition.conditions.push(operator.condition);
    } else {
      for (var j = 0; j < values.length; ++j) {
        var value = values[j];

        var subCondition = {
          operator: localOperator,
          column: column,
          values: [value]
        };

        if (localOperator === "containsElement") {
          subCondition.values.push(localSeparator);
        }

        condition.conditions.push(subCondition);
      }

      for (var j = 0; j < localWildcards.length && values.length > 0; ++j) {
        var value = localWildcards[j];

        var subCondition = {
          operator: "eq",
          column: column,
          values: [value]
        };

        condition.conditions.push(subCondition);
      }
    }

    conditions.push(condition);
  }

  for (var i = 0; i < addConditions.length; ++i) {
    var addCondition = addConditions[i];

    if (typeof addCondition.attribute === "string") {
      var column = addCondition.attribute;
      var operator = addCondition.operator;
      var localOperator;
      var localWildcards;
      var localSeparator;
      var localInferenceType;
      var localInferenceInverse;
      var localNoAutoDeselect;
      var localAutoSelect;

      if (typeof operator === "string") {
        localOperator = operator;
        localWildcards = wildcards;
        localSeparator = separator;
        localInferenceType = inferenceType;
        localInferenceInverse = inferenceInverse;
        localNoAutoDeselect = noAutoDeselect;
        localAutoSelect = autoSelect;
      } else {
        localOperator = addCondition.operator || "eq";
        localWildcards = addCondition.wildcards || wildcards;
        localSeparator = typeof addCondition.separator !== "undefined" ? addCondition.separator : separator;
        localInferenceType = addCondition.inferenceType || inferenceType;
        localInferenceInverse = typeof addCondition.inferenceInverse !== "undefined" ? addCondition.inferenceInverse : inferenceInverse;
        localNoAutoDeselect = typeof addCondition.noAutoDeselect !== "undefined" ? addCondition.noAutoDeselect : noAutoDeselect;
        localAutoSelect = typeof addCondition.autoSelect !== "undefined" ? addCondition.autoSelect : autoSelect;
      }

      // var columnDataType = dbColumns[column] ?? "STRING";
      // if (localWildcards
      //     && typeof localWildcards === "object"
      //     && localWildcards.length > 0
      //     && (columnDataType === "BOOL" || columnDataType === "INTEGER" || columnDataType === "DOUBLE")) {
      //   localWildcards = [null];
      // }

      var values = addCondition.values;
      var condition = {
        disabled: values.length === 0 || (values.length === 1 && values[0] === ""),
        operator: "or",
        conditions: []
      };

      if (addCondition.condition) {
        condition.conditions.push(addCondition.condition);
      } else {
        for (var j = 0; j < values.length; ++j) {
          var value = values[j];

          var subCondition = {
            operator: localOperator,
            column: column,
            values: [value]
          };

          if (localOperator === "containsElement") {
            subCondition.values.push(localSeparator);
          }

          condition.conditions.push(subCondition);
        }

        for (var j = 0; j < localWildcards.length && values.length > 0; ++j) {
          var value = localWildcards[j];

          var subCondition = {
            operator: "eq",
            column: column,
            values: [value]
          };

          condition.conditions.push(subCondition);
        }
      }

      conditions.push(condition);
    } else {
      addCondition.noAutoDeactivate = true;

      conditions.push(addCondition);
    }
  }

  var queryObj = {
    database: database,
    from: tableName,
    select: dbAttributes.map(x => x.column),
    where: {
      operator: "and",
      conditions: conditions
    },
    callback: function (status, vtResult, index) {
      var attribute = dbAttributes[index].attribute;
      var inferenceInfo = inferenceMap[attribute];
      var values = vtResult.values;

      if (typeof inferenceInfo.separator !== "undefined") {
        var allValues = [];

        for (var i = 0; i < values.length; ++i) {
          var value = values[i];
          var rowValues;

          if (typeof value === "string") {
            rowValues = value.split(inferenceInfo.separator);
          } else {
            rowValues = [value];
          }

          for (var j = 0; j < rowValues.length; ++j) {
            if (rowValues[j] !== "") {
              allValues.push(rowValues[j]);
            }
          }
        }

        values = allValues;
      }

      if (inferenceInfo.wildcards && inferenceInfo.wildcards.length > 0) {
        for (var i = 0; i < values.length; ++i) {
          var value = values[i];

          if (VariantTables.Util.indexOf(inferenceInfo.wildcards, value) >= 0) {
            values = ecl
              .ECL_GetAllAttributeValues(attribute)
              .map(x => (x === true || x === false) ? (x ? 1 : 0) : x);
            break;
          }
        }
      }

      if (status.error) {
        result.error = true;
      }

      result.data[attribute] = values;

      me._logger.trace("restricting attribute", attribute,
        "( inferenceType:", inferenceInfo.type,
        "; inferenceInverse:", inferenceInfo.inverse,
        "; wildcards:", ((typeof inferenceInfo.wildcards === "object") ? VariantTables.Util.getArrayAsString(inferenceInfo.wildcards) : ""),
        "; separator:", inferenceInfo.separator,
        "; noAutoDeselect:", inferenceInfo.noAutoDeselect,
        "; autoSelect:", inferenceInfo.autoSelect, ")");
      inference.restrictAttribute(attribute, values, inferenceInfo);
    }
  };

  query.query(queryObj);

  deltaTime = VariantTables.Util.getTime() - deltaTime;

  this._logger.trace("simple inference time:", deltaTime, "ms");

  return result;
};

VariantTables.SimpleInference.prototype._getDatabaseAttributes = function (database, tableName, attrMap) {
  var columns = ecl.ECL_DatabaseGetTableColumns(database, tableName) ?? [];
  var invAttrMap = VariantTables.Util.getInverseUpperCaseMap(attrMap);
  var allAttributes = ecl.ECL_GetAllAttributes();
  var mapUpper = Object.fromEntries(allAttributes.map(x => [x.toUpperCase(), x]));
  var results = [];

  for (var i = 0; i < columns.length; ++i) {
    var column = columns[i].Name;
    var dataType = columns[i].DataType.toUpperCase();

    if (column in attrMap && attrMap[column] === null) {
      continue;
    }

    var columnUpper = column.toUpperCase();

    if (columnUpper in invAttrMap) {
      results.push({ column: column, attribute: invAttrMap[columnUpper], dataType: dataType });
    } else if (columnUpper in mapUpper) {
      results.push({ column: column, attribute: mapUpper[columnUpper], dataType: dataType });
    }
  }

  return results;
};

VariantTables.SimpleInference.prototype._getDatabaseColumns = function (database, tableName) {
  var columns = ecl.ECL_DatabaseGetTableColumns(database, tableName) ?? [];
  var columnMap = {};

  for (var i = 0; i < columns.length; ++i) {
    columnMap[columns[i].Name] = columns[i].DataType.toUpperCase();
  }

  return columnMap;
};
