//@legacy
//#mode(full-ecma)


//@import "Base.misc.js"

//@import "SD/MLFB.MOTOX.SD.GearedMotor.js"

//@import "SD/MLFB.MOTOX.SD.Motor.js"

//@import "SD/MLFB.MOTOX.SD.Gear.js"

//@import "SD/MLFB.MOTOX.SD.Brake.js"

//@import "SD/MLFB.MOTOX.SD.Encoder.js"


/**
 * @category   MOTOX MLFB generation
 * @package    product logic
 * @copyright  Copyright (c) 2011-2012 Plan Software GmbH (http://www.plansoft.de)
 */

/**
 * @constructor
 * prevent to add a short description twice, test if a short description is already present, add it if not
 * 
 * @param strSD - short description to test
 * @param aShortDescriptions - array with short descriptions to modify
 *
 * @return array - possibly modified array of short descriptions
 */
 
function preventDoubledShortDescription( strSD, aShortDescriptions )
{
    if ( strSD !== '' )
    {
        // prevent to add a short descriptions twice (they could have been handled automatically)
        strSD = '+' + strSD;
        if ( containsString( aShortDescriptions, strSD, 0 ) < 0 )
            aShortDescriptions.push( strSD );
    }
    
    return( aShortDescriptions );
}

function preventDoubledShortDescriptionMultiple( strSD, aShortDescriptions )
{
    if ( strSD !== '' )
    {
        //Multiple KA's in a String
        var elements = stringtoarray(strSD, "TEMP.SPECIFICATIONMLFB",",");
        for(var i=0;i<elements;i++)
        {
            var path = "TEMP.SPECIFICATIONMLFB["+i+"]";
            var Temp = getstring("VALUE", path);
            
            strSD = '+' + Temp;
            
            // prevent to add a short descriptions twice (they could have been handled automatically)
            if ( containsString( aShortDescriptions, strSD, 0 ) < 0 )
            aShortDescriptions.push( strSD );
            
        }
        
        
    }
    
    return( aShortDescriptions );
}


/**
 * @constructor
 * get the short descriptions for various gear options, which can be computed automatically by a simple lookup in a database tables OPTION_VALUES in 
 * the database MOTOXKurzangaben.db
 * 
 * @param oGear - geared motor object
 *
 * @return array - each index contains a string with one short description (may be empty)
 */
function getStandardShortDescriptions( oGear )
{	
    //--------------------------------------------------------------------------------------------------------
    // that's the heavy part: the exceptions from the rule below
    //--------------------------------------------------------------------------------------------------------

    // stage 1:
    // this way of generating short descriptions is not used for the first stage of implementation
    // reason: as long as the 'MLFB bible for SimoGear' is not ready and the MLFB data is not present or doesn't match the product logic, there's no easy way to 
    // develop the necessary database tables for MOTOXKurzangaben.db
    // Instead, all short descriptions are generated by code
    //
    // stage 2:
    // still todo: refactor the short description generation so that the table approach can be used again
    //
    
    //--------------------------------------------------------------------------------------------------------
    // that's the simple part: the standard lookup of configuration attributes and values in the database
    //--------------------------------------------------------------------------------------------------------
    var aShortDescriptions = [];
    
    if ( oGear.oConfig )
    {
        var aStandard = MLFB.getShortDescriptionsFromDatabase( oGear, oExceptions, '2KJ4_KURZANGABEN_DB' );
        if ( aStandard.length > 0 )
            aShortDescriptions = aShortDescriptions.concat( aStandard );
    }
    
    return( aShortDescriptions );
}


/**
 * @constructor
 * get short descriptions for the geared motor and the geared motor configuration
 * 
 * @param oGear - geared motor object
 *
 *
 * @return array - each index contains a string with one short description (may be empty)
 */
function getShortDescriptions( oGear )
{
    // get descriptions
    var aCommonDescriptions = getGearedMotorDescriptions( oGear );
    
    
    var aGearDescriptions = getGearDescriptions( oGear );
    
    
    var aMotorDescriptions = [];
    var aAdaperDescriptions =[];
    var aMotorG110MDescriptions = [];
    var aBrakeDescriptions = [];
    var aEncoderDescriptions = [];
     
    if(ECL_GetValue ("ECSI_2KJ_TYPE_2KJ4") == "GETRIEBEMOTOR")
    {
        aMotorDescriptions = getMotorDescriptions( oGear );
        aBrakeDescriptions = getBrakeDescriptions( oGear );
        aEncoderDescriptions = getEncoderDescriptions( oGear );
    
    }
    else if(ECL_GetValue ("ECSI_2KJ_TYPE_2KJ4") == "GETRIEBE.ADAPTER")
        aAdaperDescriptions = getAdapterDescriptions( oGear );
    
    
    // due to a bug in the EasyScript interpreter, we cannot nest the Array.concat calls
    var aSD = aCommonDescriptions.concat( aGearDescriptions );
    
    
    if ( aMotorDescriptions.length > 0 )  {
        aSD = aSD.concat( aMotorDescriptions );
    }
    if ( aAdaperDescriptions.length > 0 ) {
        aSD = aSD.concat( aAdaperDescriptions );
    }
    if ( aBrakeDescriptions.length > 0 )  {
        aSD = aSD.concat( aBrakeDescriptions );
    }
    if ( aEncoderDescriptions.length > 0 ){
        aSD = aSD.concat( aEncoderDescriptions );
    }
    
    
    // for special remarks (Y99)
    //if ( ( oGear.oConfig.strNotes != '' ) && ( aShortDescriptions.indexOf( '+Y99' ) < 0 ) )
    
    /*
    if ( ( oGear.oConfig.strNotes != '' ) && ( Base.indexOf( aShortDescriptions, '+Y99' ) < 0 ) )	// necessary due to a bug in EasyScript (see trac ticket #70)
        aSD.push( '+Y99' );
    */
    
    return( aSD );
    
}


/**
 * @constructor
 * three filtering and modifying functions
 * 1.) some short descriptions that are not price-relevant may be filtered out
 * 2.) a surplus to 32 short descriptions must be combined into a single Y00-description
 * 3.) multiple Y00 descritptions must be coalesced into a single one
 * 
 * @param oGear - geared motor object that is described with the MLFB
 * @param strMLFB - base MLFB of the gear
 * @param aShortDescriptions - short descriptions to filter
 *
 * @return array - each index contains a string with one short description (may be empty)
 */
function filterShortDescriptions( oGear, strMLFB, aShortDescriptions )
{
    // some short descriptions that are not price-relevant have to be filtered out		
    //var aDescriptions = _filterShortDescriptions( oGear, strMLFB, aShortDescriptions );
    
    // if we have more than 32 short descriptions, we must coalesce some descriptions into a single Y00-description
    /*
    if ( aDescriptions.length > 32 )
        aDescriptions = _coalesceTo32Descriptions( strMLFB, aDescriptions );
    */
    // if we have multiple Y00-descriptions, we have to coalesce them into a single Y00 description
    aDescriptions = _coalesceY00( aShortDescriptions );

    return( aDescriptions );
}


/**
 * @constructor
 * multiple Y00 short descriptions have to be coalesced into a single Y00 description
 * 
 * @param aShortDescriptions - short descriptions to coalesce
 *
 * @return array - coalesced short descriptions; each index contains a string with one short description (may be empty)
 */
function _coalesceY00( aShortDescriptions )
{
    if ( aShortDescriptions.length === 0 )
        return( aShortDescriptions );
    
    var aDescriptions = [];
    var nCountSD = aShortDescriptions.length;
    var strSD = '', strY00 = '';
    for( var i = 0; i < nCountSD; ++i )
    {
        strSD = aShortDescriptions[i];
        
        if ( strSD.indexOf( 'Y00' ) > -1 )
        {
            // collect all Y00 in a single string
            if ( strY00 === '' )
            {
                strY00 = trimright( strSD, '\'' );
                strY00 += "|";
            }
            else
            {
                strY00 += trimright( replace( strSD, '+Y00\'', '' ), '\'' );
                strY00 += "|";
            }
        }
        else aDescriptions.push( strSD );
    }
    if ( strY00 !== '' )
    {
        strY00=trimright(strY00,"|");
        aDescriptions.push( strY00 + '\'' );
    }
    return( aDescriptions );
}



/**
 * @constructor
 * if we have more than 32 short descriptions, some of them have to be coalesced into a single Y..-short description
 * 
 * @param strMLFB - base MLFB to this short description
 * @param aShortDescriptions - short descriptions to coalesce
 *
 * @return array - coalesced short descriptions; each index contains a string with one short description (may be empty)
 */
function _coalesceTo32Descriptions( strMLFB, aShortDescriptions )
{
    if ( aShortDescriptions.length <= 32 )
        return( aShortDescriptions );
    
    // split descriptions in those that can be placed in the Y00 descriptions (all descriptions with a price of 0 that are not delivery time relevant) and those that
    // must be present in the short descriptions array
    var aY00Descriptions = [], aMustHaveDescriptions = [];
    var nCountSD = aShortDescriptions.length;
    var strSD = '', strSDPart = '';
    for( var i = 0; i < nCountSD; ++i )
    {
        // there are two layers of possibilities that must be tested for a short description:
        // 1.) the database table MLFB_OPTIONEN contains a column KA_GEN_KNZ that determines if a short description may be part of a Y00
        // 2.) if a short description may be part of Y00, we must query its price and its delivery time; the description can only be part of Y00 if its current price
        //     and current delivery time is neclectable
        strSD = aShortDescriptions[i];
        strSDPart = strSD.substring( 1, 4 );
        
        var bValidKA = true;
        if(strSDPart.left(1) == "P" || strSDPart.left(1) == "N" || strSDPart.left(1) == "H")
            bValidKA = false;
        
        var oData = Base.getPMDDataForShortDescription( strMLFB, strSDPart );
        if ( parseFloat(oData.strPrice) === 0 && parseFloat(oData.strDeliveryTime) <= 15 && parseFloat(oData.strWeight) === 0 && bValidKA)
            aY00Descriptions.push( strSD );
        else aMustHaveDescriptions.push( strSD );
        
    }
    
    aY00Descriptions.sort();
    var nCountSDY00 = aY00Descriptions.length;
    var strY00Opt = '';
    for ( var i = 0; i < nCountSDY00; ++i )
    {
        if ( aMustHaveDescriptions.length < 31 )	// remember the additional Y00
            aMustHaveDescriptions.push( aY00Descriptions[i] );
        else strY00Opt += aY00Descriptions[i];
    }
    if ( strY00Opt !== '' )
        aMustHaveDescriptions.push( "+Y00'*OPT@" + trimleft( strY00Opt, '+' ) + "*'" );
    
    return( aMustHaveDescriptions );
}


/**
 * @constructor
 * some short descriptions that are not price-relevant have to be filtered out
 * this is also the right place to temporarily disable some short descriptions in the final MLFB without the need to eliminate the 
 * option-relevant attributes 
 * 
 * @param oGear - geared motor object that is described with the MLFB
 * @param strMLFB - base MLFB of the gear
 * @param aShortDescriptions - short descriptions to filter
 *
 * @return array - filtered short descriptions; each index contains a string with one short description (may be empty)
 */
function _filterShortDescriptions( oGear, strMLFB, aShortDescriptions )
{
    // test if a short description may be eliminated based on its PMD data
    var strRegion = "TUB";
    var _mayEliminate = function( oPMDData )
    {
        var minLieferzeit;

        if(strRegion=="CHINA")
        {
            minLieferzeit=20;
        }
        else
        {
            minLieferzeit=25;
        }
        
        if ( !oPMDData )		// no PMD data means no comparision posible - we'd better leave this short description in the MLFB
            return( false );
        var dPrice = oPMDData.strPrice - 0;
        var dTime = oPMDData.strDeliveryTime - 0;
        return( ( dPrice == 0 ) && ( dTime <= minLieferzeit ) );		// no price and minimal delivery time means: can be eliminated
    }
    
    var nLen = aShortDescriptions.length;
    var aDescriptions = [];
    var bAdd = false, oPMD, strSD;
    for ( var i = 0; i < nLen; ++i )
    {
        // filtering means testing some descriptions if they have prices or higher delivery times or are set to the default for the current gear
        strSD = aShortDescriptions[i]
        bAdd = true;
        
        if ( ( strSD === '+G09' ) || ( strSD === '+G10' ) )
        {
            // figur for torque support
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current figur is the default for this gear
                var strDefaultFigur = Product.MOTOX.Config.Design.getCombo_Flanschfigur_Default_Value( oGear.oConfig, oGear, oMOTOXHandler );
                if ( strDefaultFigur === oGear.oConfig.strFigur )
                    bAdd = false;
            }
        }
        else if ( ( strSD === '+G15' ) || ( strSD === '+G16' ) || ( strSD === '+G17' ) )
        {
            // housing material
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current bearing is the default for this gear
                var SDDefaultGHW = MLFB.MOTOX.SD.getShortDescription( 'EGGEHWS', Product.MOTOX.Config.OtherGearOptions.getCombo_Gehaeusematerial_Default(oGear.oConfig,oGear));
                if ( SDDefaultGHW === strSD )
                    bAdd = false;
            }
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
        }
        else if ( ( strSD === '+G19' ) || ( strSD === '+G20' ) || ( strSD === '+G21' ) )
        {
            // output shaft bearing
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current bearing is the default for this gear
                var strDefaultBearing = Product.MOTOX.Config.OutputShaft.getCombo_Abtriebswellenlagerung_DefaultValue( oGear.oConfig, oGear );
                if ( strDefaultBearing === oGear.oConfig.strOutputBearing )
                    bAdd = false;
            }
        }
        else if ( ( strSD === '+G22' ) || ( strSD === '+G23' ) || ( strSD === '+G24' ) || ( strSD === '+G26' ) )
        {
            // output shaft sealing
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current sealing is the default for this gear
                var strDefaultSealing = Product.MOTOX.Config.Lubrication.getCombo_Abtriebswellenabdichtung_DefaultValue( oGear.oConfig, oGear, oMOTOXHandler );
                if ( strDefaultSealing === oGear.oConfig.strOutputShaftSeal )
                    bAdd = false;
            }
        }
        else if ( ( strSD === '+G32' ) || ( strSD === '+G33' ) || ( strSD === '+G34' ) )
        {
            // oil level control
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current oil level control is the default for this gear
                var strDefaultControl = Product.MOTOX.Config.Oil.getCombo_Oelstandskontrolle_DefaultValue( oGear.oConfig, oGear, oMOTOXHandler );
                
                if(strRegion=="CHINA")
                {
                    if (oGear.oConfig.strOilLevelControl == "Oelstands-Kontrollschraube")
                        bAdd = false;
                }
                else
                {
                    if ( strDefaultControl === oGear.oConfig.strOilLevelControl )
                        bAdd = false;
                }
            }
        }
        else if ( ( strSD === '+G43' ) || ( strSD === '+G45' ) || ( strSD === '+G47' ) || ( strSD === '+G48' ) || ( strSD === '+G49' ) )
        {
            // gear breather
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current oil breather is the default for this gear
                var strDefaultBreather = Product.MOTOX.Config.Oil.getCombo_Entlueftung_DefaultValue( oGear.oConfig, oGear, oMOTOXHandler );
                if ( strDefaultBreather === oGear.oConfig.strGearboxBreather )
                    bAdd = false;
            }
            
            if( (oGear.GearData.EGBEZTYP == "-" || oGear.GearData.EGBEZTYP == "F" || oGear.GearData.EGBEZTYP == "B") && ( oGear.GearData.EGBEZDIM == "19" || oGear.GearData.EGBEZDIM == "29") && strSD == "+G45")
                bAdd = true;
        }
        else if ( ( strSD === '+G51' ) || ( strSD === '+G52' ) || ( strSD === '+G53' ) || ( strSD === '+G54' ) || ( strSD === '+G55' ) )
        {
            // oil drain
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current oil drain is the default for this gear
                var strDefaultDrain = Product.MOTOX.Config.Oil.getCombo_Oelablass_DefaultValue( oGear.oConfig, oGear, oMOTOXHandler );
                if ( strDefaultDrain === oGear.oConfig.strOilDrain )
                    bAdd = false;
            }
        }
        else if ( ( strSD === '+G58' ) || ( strSD === '+G59' ) || ( strSD === '+G60' ) )
        {
            // hollow shaft cover
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
            {
                // can be eliminated if the current cover is the default for this gear
                var strDefaultCover = Product.MOTOX.Config.Design.getCombo_Hohlwellenabdeckung_DefaultValue( oGear.oConfig, oGear, oMOTOXHandler );
                if ( strDefaultCover === oGear.oConfig.strHollowShaftCover )
                    bAdd = false;
            }
        }
        else if ( strSD === '+N53' )
        {
            // external ground screw can be elminated if it has no price or delivery time
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
                bAdd = false;
        }
        else if ( strSD === '+C00' )
        {
            // brake design 'normal' can be elminated if it has no price or delivery time
            oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD.substring( 1 ) );
            if ( _mayEliminate( oPMD ) )
                bAdd = false;
        }
        
        if ( bAdd )
            aDescriptions.push( strSD );
    }
    
    return( aDescriptions );
}

function filterShortDescriptionsforConfigOptions( oMOTOX, oDlg, strMLFB, strSD )
{
    var minLieferzeit;
    
    var oMOTOXHandler 	= getDriveTrainHandler().getMOTOXHandler();
    var strRegion = oMOTOXHandler.getRegion();
    if(strRegion=="CHINA")
    {
        minLieferzeit=20;
    }
    else
    {
        minLieferzeit=25;
    }
    
    
    // test if a short description may be eliminated based on its PMD data
    var _mayEliminate = function( oPMDData )
    {
        if ( !oPMDData )		// no PMD data means no comparision posible - we'd better leave this short description in the MLFB
            return( false );
        var dPrice = oPMDData.strPrice - 0;
        var dTime = oPMDData.strDeliveryTime - 0;
        return( ( dPrice == 0 ) && ( dTime <= minLieferzeit ) );		// no price and minimal delivery time means: can be eliminated
    }
    
    var bAdd = true, oPMD;
    
    // filtering means testing some descriptions if they have prices or higher delivery times or are set to the default for the current gear
    if ( ( strSD === 'G15' ) || ( strSD === 'G16' ) || ( strSD === 'G17' ) )
    {
        // Housing
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current bearing is the default for this gear
            var SDDefaultGHW = MLFB.MOTOX.SD.getShortDescription( 'EGGEHWS', Product.MOTOX.Config.OtherGearOptions.getCombo_Gehaeusematerial_Default(oDlg,oMOTOX));
            if ( SDDefaultGHW === strSD )
                bAdd = false;
        }
    }
    if ( ( strSD === 'G19' ) || ( strSD === 'G20' ) || ( strSD === 'G21' ) )
    {
        // output shaft bearing
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current bearing is the default for this gear
            var SDDefaultBearing = MLFB.MOTOX.SD.getShortDescription( 'EGABALAG', Product.MOTOX.Config.OutputShaft.getCombo_Abtriebswellenlagerung_DefaultValue( oDlg, oMOTOX ));
            if ( SDDefaultBearing === strSD )
                bAdd = false;
        }
    }
    else if ( ( strSD === 'G22' ) || ( strSD === 'G23' ) || ( strSD === 'G24' ) || ( strSD === 'G25' ) || ( strSD === 'G26' ) || ( strSD === 'G27' ) )
    {
        // output shaft sealing
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current sealing is the default for this gear
            var SDDefaultSealing = MLFB.MOTOX.SD.getShortDescription( 'EGDICHTART', Product.MOTOX.Config.Lubrication.getCombo_Abtriebswellenabdichtung_DefaultValue( oDlg, oMOTOX, oMOTOXHandler ));
            if ( SDDefaultSealing === strSD )
                bAdd = false;
        }
    }
    else if ( ( strSD === 'G32' ) || ( strSD === 'G33' ) || ( strSD === 'G34' ) )
    {
        // oil level control
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current oil level control is the default for this gear
            var SDDefaultControl = MLFB.MOTOX.SD.getShortDescription('EGGEHOLK', Product.MOTOX.Config.Oil.getCombo_Oelstandskontrolle_DefaultValue( oDlg, oMOTOX, oMOTOXHandler ));
            
            if(strRegion=="CHINA")
            {
                if (strSD== "G33")
                    bAdd = false;
            }
            else
            {
                if ( SDDefaultControl === strSD )
                    bAdd = false;
            }
        }
    }
    else if ( ( strSD === 'G43' ) || ( strSD === 'G45' ) || ( strSD === 'G47' ) || ( strSD === 'G48' ) || ( strSD === 'G49' ) )
    {
        // gear breather
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current oil breather is the default for this gear
            var SDDefaultBreather = MLFB.MOTOX.SD.getShortDescription('EGGEHENT', Product.MOTOX.Config.Oil.getCombo_Entlueftung_DefaultValue( oDlg, oMOTOX, oMOTOXHandler ));
            if ( SDDefaultBreather === strSD )
                bAdd = false;
        }
        
        if( (oMOTOX.GearData.EGBEZTYP == "-" || oMOTOX.GearData.EGBEZTYP == "F" || oMOTOX.GearData.EGBEZTYP == "B") && ( oMOTOX.GearData.EGBEZDIM == "19" || oMOTOX.GearData.EGBEZDIM == "29") && strSD == "G45")
            bAdd = true;
    }
    else if ( ( strSD === 'G51' ) || ( strSD === 'G52' ) || ( strSD === 'G53' ) || ( strSD === 'G54' ) || ( strSD === 'G55' ) )
    {
        // oil drain
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current oil drain is the default for this gear
            var SDDefaultDrain = MLFB.MOTOX.SD.getShortDescription( 'EGGEHOLA', Product.MOTOX.Config.Oil.getCombo_Oelablass_DefaultValue( oDlg, oMOTOX, oMOTOXHandler ));
            if ( SDDefaultDrain === strSD )
                bAdd = false;
        }
    }
    else if ( ( strSD === 'G58' ) || ( strSD === 'G59' ) || ( strSD === 'G60' ) )
    {
        // hollow shaft cover
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
        {
            // can be eliminated if the current cover is the default for this gear
            var SDDefaultCover = MLFB.MOTOX.SD.getShortDescription( 'EGABABDW', Product.MOTOX.Config.Design.getCombo_Hohlwellenabdeckung_DefaultValue( oDlg, oMOTOX, oMOTOXHandler ));
            if ( SDDefaultCover === strSD )
                bAdd = false;
        }
    }
    else if ( strSD === 'N53' )
    {
        // external ground screw can be elminated if it has no price or delivery time
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
            bAdd = false;
    }
    else if ( strSD === 'C00' )
    {
        // brake design 'normal' can be elminated if it has no price or delivery time
        oPMD = Base.getPMDDataForShortDescription( strMLFB, strSD );
        if ( _mayEliminate( oPMD ) )
            bAdd = false;
    }
    
    return( bAdd );
}

