//@legacy
//#mode(full-ecma)

/**
 * @category   MOTOX MLFB generation
 * @package    product logic
 * @copyright  Copyright (c) 2012 Plan Software GmbH (http://www.plansoft.de)
 */
//@import "../../2KJ4_CONFIG/utils.js"


/**
 * @constructor
 * get short descriptions for the motor attributes
 * 
 * @param oGear - geared motor object
 *
 * @return array - short descriptions for the motor attributes,
 *                 each index contains a string with one short description (may be empty)
 */
function getMotorDescriptions( oGear )
{
    var aShortDescriptions = [];
    
    _addElectricDescriptions( oGear, aShortDescriptions );
    _addMechanicalDescriptions( oGear, aShortDescriptions );
    _addDescriptionsForOtherMotorOptions( oGear, aShortDescriptions );
    
    if(ECL_GetValue("ECSI_ATEX_KENNER_2KJ4") == "ATEX_INTEGRATED")
        _addATEXDescriptions( oGear, aShortDescriptions );

    return( aShortDescriptions );
}

function getAdapterDescriptions( oGear )
{
    var aShortDescriptions = [];
    
    _addAdapterKADescriptions( oGear, aShortDescriptions );
    
    _addAdapterOptionDescriptions( oGear, aShortDescriptions );

    _addAdapterKUOptionDescriptions( oGear, aShortDescriptions );
    
    _addATEXDescriptions_Adapter( oGear, aShortDescriptions );
    
    return( aShortDescriptions );
}

/**
 * @constructor
 * add short descriptions for the motor ATEX options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for motor ATEX options descriptions
 */
function _addATEXDescriptions_Adapter( oGear, aShortDescriptions )
{
    // ATEX Kennzeichnung Getriebe bei Anbau 1MB1
    var strATEX_Adapter = ECL_GetValue("ECSI_ATEX_KENNER_2KJ4");
    
    if(strATEX_Adapter=="ATEX_ADAPTER")
    {
        var strAtexZone = ECL_GetValue("ECSI_ATEX_ZONE_2KJ4");
        var strAtexExplosionsGruppe = ECL_GetValue("ECSI_ATEX_EGRUPPE_2KJ4");
        var strAtexTKlasse = ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4");
        
        //Zone
        addShortDescription( aShortDescriptions, 'ATEXZONE', strAtexZone );
        
        var strOelstandskontrolle="";
        if(ECL_GetValue("OELSTANDSKONTROLLE_2KJ4")=="")
        {
            //Wenn wir aus dem Parser direkt kommen, hat die ölstandkontrolle noch keinen Wert
            var strSD_Oelstandskontrolle = getSDFromGroup( aShortDescriptions, 'OELSTANDKONTROLLE' );
            
            if(strSD_Oelstandskontrolle=="")
                strSD_Oelstandskontrolle="G32";
                
            var strSelect="SELECT DISTINCT PROPERTY_VALUE FROM OPTION_VALUES WHERE KURZANGABE='"+strSD_Oelstandskontrolle+"' ";
            if ( dbselect( "2KJ4_KURZANGABEN_DB", strSelect, "2KJ4_KURZANGABEN_DB" ) > 0 )
            {
                strOelstandskontrolle = dbgetstring("2KJ4_KURZANGABEN_DB",'PROPERTY_VALUE');
            }
            else
            {
                logtraceln("Fehler bei Bestimmung der Ölstandskontrolle, wenn wir aus dem Parser kommen! SQL="+strSelect);
            }
        }
        else
        {
            strOelstandskontrolle=ECL_GetValue("OELSTANDSKONTROLLE_2KJ4");
        }
        
        //Getriebe
        var strGetriebeKennzeichnungATEX="";
        var strDBName = "2KJ4_DB"; 
        var strDBResult = "ATEX_GETRIEBE_KENNZEICHNUNG_MLFB";
        
        var strSelect = "";
        strSelect = "SELECT DISTINCT ATEXKNZGETR FROM ATEXKNZGETR WHERE ";
        strSelect += "ZONE = '"+strAtexZone+"' AND ATEXKATEGORIE = '"+strAtexExplosionsGruppe+"'  AND EGGEHOLK = '"+strOelstandskontrolle+"' AND AKTIV='1' AND MOTOX_N='0' AND REGION='S' AND SONDER='-' ";
    
        if ( dbselect( strDBName, strSelect, strDBResult ) > 0 )
        {
            strGetriebeKennzeichnungATEX = dbgetstring(strDBResult,'ATEXKNZGETR');
        }
        else
        {
            logtraceln("Fehler bei Bestimmung der ATEX Kennzeichnung des Getriebes! SQL="+strSelect);
        }
        
        addShortDescription( aShortDescriptions, 'ATEXKNZGETR', strGetriebeKennzeichnungATEX );
        
        //ATEX Oberflächentemperatur
        if(ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4")=="T4")
            aShortDescriptions.push('+K78');
        else if(ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4")=="120")
            aShortDescriptions.push('+K75');
        
    }
    
    
}

/**
 * @constructor
 * add short descriptions for the motor ATEX options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for motor ATEX options descriptions
 */
function _addATEXDescriptions( oGear, aShortDescriptions )
{
    var strAtexZone = ECL_GetValue("ATEX_ZONE_2KJ4");
    var strAtexExplosionsGruppe = ECL_GetValue("ATEX_EXPLOSIONSGRUPPE_2KJ4");
    var strAtexTKlasse = ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4");
    
    //Zone
    addShortDescription( aShortDescriptions, 'ATEXZONE', strAtexZone );
    
    var strOelstandskontrolle="";
    if(ECL_GetValue("OELSTANDSKONTROLLE_2KJ4")=="")
    {
        //Wenn wir aus dem Parser direkt kommen, hat die ölstandkontrolle noch keinen Wert
        var strSD_Oelstandskontrolle = getSDFromGroup( aShortDescriptions, 'OELSTANDKONTROLLE' );
        
        if(strSD_Oelstandskontrolle=="")
            strSD_Oelstandskontrolle="G32";
            
        var strSelect="SELECT DISTINCT PROPERTY_VALUE FROM OPTION_VALUES WHERE KURZANGABE='"+strSD_Oelstandskontrolle+"' ";
        if ( dbselect( "2KJ4_KURZANGABEN_DB", strSelect, "2KJ4_KURZANGABEN_DB" ) > 0 )
        {
            strOelstandskontrolle = dbgetstring("2KJ4_KURZANGABEN_DB",'PROPERTY_VALUE');
        }
        else
        {
            logtraceln("Fehler bei Bestimmung der Ölstandskontrolle, wenn wir aus dem Parser kommen! SQL="+strSelect);
        }
    }
    else
    {
        strOelstandskontrolle=ECL_GetValue("OELSTANDSKONTROLLE_2KJ4");
    }
    
    //Getriebe
    var strGetriebeKennzeichnungATEX="";
    var strDBName = "2KJ4_DB"; 
    var strDBResult = "ATEX_GETRIEBE_KENNZEICHNUNG_MLFB";
    
    var strSelect = "";
    strSelect = "SELECT DISTINCT ATEXKNZGETR FROM ATEXKNZGETR WHERE ";
    strSelect += "ZONE = '"+strAtexZone+"' AND ATEXKATEGORIE = '"+strAtexExplosionsGruppe+"'  AND EGGEHOLK = '"+strOelstandskontrolle+"' AND AKTIV='1' AND MOTOX_N='0' AND REGION='S' AND SONDER='-' ";

    if ( dbselect( strDBName, strSelect, strDBResult ) > 0 )
    {
        strGetriebeKennzeichnungATEX = dbgetstring(strDBResult,'ATEXKNZGETR');
    }
    else
    {
        logtraceln("Fehler bei Bestimmung der ATEX Kennzeichnung des Getriebes! SQL="+strSelect);
    }
    
    addShortDescription( aShortDescriptions, 'ATEXKNZGETR', strGetriebeKennzeichnungATEX );
    
    //ATEX Oberflächentemperatur
    if(ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4")=="T4")
        aShortDescriptions.push('+K78');
    else if(ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4")=="T3")
        aShortDescriptions.push('+K77');
    else if(ECL_GetValue("ECSI_ATEX_TCLASS_2KJ4")=="120")
        aShortDescriptions.push('+K75');
        
}


/**
 * @constructor
 * add short descriptions for the electric motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for electric motor options descriptions
 */
function _addElectricDescriptions( oGear, aShortDescriptions )
{
    // motor efficiency class is already encoded in the MLFB trunk (digit 12)
    
    // varying motor power is computed with the trunk MLFB-part for the motor
    
    // poles
    addShortDescription( aShortDescriptions, 'POLZAHL', oGear.oRow.POLZAHL );
    
    // electrical specification is computed with the trunk MLFB-part for the motor
    
    // motor voltage is encoded in the trunk MLFB-part for the motor (and in its necessary short description)

    // thermical motorprotection
    var strProtection = ECL_GetValue("MOTORSCHUTZ_2KJ4");
    if ( strProtection !== 'ohne' )
        addShortDescription( aShortDescriptions, 'WIELART', strProtection );		
    
    // anti-condensation heating
    var strHeating = ECL_GetValue("STILLSTANDSHEIZUNG_2KJ4");
    if ( strHeating !== 'ohne' )
        addShortDescription( aShortDescriptions, 'WIELHEIZ', 'mit', 'WIELHEIZSPG', strHeating );		

    // winding temperature class
    var strClass = ECL_GetValue("WAERMEKLASSE_2KJ4");
    if ( strClass !== 'F' )
        addShortDescription( aShortDescriptions, 'MOTWK', strClass );		
    
    // reinforced insulation
    /*
    if ( oGear.oConfig.bReinforcedInsulation )
        MLFB.MOTOX.SD.addShortDescription( aShortDescriptions, 'VERSTISO', 'ja' );
    */
    
    // increased protection for special environments
    if ( ECL_ValueSelected("SAEURESCHUTZ_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'MOFSSCHUTZWI', 'ja' );
        
    if ( ECL_GetValue("VERSANDSCHALTUNG_2KJ4")!="-")
    {
        if(ECL_GetValue("VERSANDSCHALTUNG_2KJ4")=="Dreieck")
            addShortDescription( aShortDescriptions, 'MOTSHLT_VSHLT_AWAHL', 'D' );	
        else if(ECL_GetValue("VERSANDSCHALTUNG_2KJ4")=="Stern")
            addShortDescription( aShortDescriptions, 'MOTSHLT_VSHLT_AWAHL', 'Y' );	
    }
}


/**
 * @constructor
 * add short descriptions for the mechanical motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for mechanical motor options descriptions
 */
function _addMechanicalDescriptions( oGear, aShortDescriptions )
{
    // motor protection
    addShortDescription( aShortDescriptions, 'SCHUTZART', ECL_GetValue("SCHUTZART_2KJ4") );

    // mechanical venting
    var strVenting = ECL_GetValue("BELUEFTUNG_2KJ4");
    if ( strVenting !== 'Standard' )
    {
        if ( strVenting === 'Schwungradluefter' )
            addShortDescription( aShortDescriptions, 'MOTSWRD', 'I', 'BELLUEFT', 'Schwungradluefter' );		
        else if ( strVenting === 'Zugeordneter Luefter' )
        {
            addShortDescription( aShortDescriptions, 'BELART', 'F' );
            
            // there are no additional short descriptions for protection and voltage of 'venting with foreign blower'
        }
        else 
            addShortDescription( aShortDescriptions, 'MOTSWRD', '-', 'BELLUEFT', strVenting );	
    }
    
    // terminal box position
    addShortDescription( aShortDescriptions, 'KKLAGE', ECL_GetValue("KLEMMKASTENLAGE_2KJ4") );
    
    //Tabelle MOTERD
    //Bei Vorgeschrieben (2), keine KA ausgeben
    
    var strMOEINKL = oGear.oRow.MOT_BZ_LANG;
    strMOEINKL = strMOEINKL.left(3);
    if(strMOEINKL.right(1) == "N")
        strMOEINKL = "N";
    else if(strMOEINKL.right(1) == "S")
        strMOEINKL = "S";
    else
        strMOEINKL = "-";
    
    var strSelect = "SELECT DISTINCT MOTERD FROM MOTERD WHERE ";
    
    strSelect = strSelect + "MOTTYP='"+oGear.oRow.MOTTYP+"' AND ";
    strSelect = strSelect + "MOTDIM='"+oGear.oRow.MOTDIM+"' AND ";
    strSelect = strSelect + "MOEINKL='"+strMOEINKL+"' AND ";
    strSelect = strSelect + "M3 = '"+oGear.oRow.M3+"' AND ";
    strSelect = strSelect + "GM3 = '"+oGear.oRow.GM3+"' AND ";
    strSelect = strSelect + "REGION = 'CE' AND ";
    strSelect = strSelect + "G115M='0' AND G115D='0' ";
        
    strSelect = strSelect + "AND CDROM_REL = '1' ";
    strSelect = strSelect + "AND AKTIV = '1' ";
    
    var aResult = ExecSQL_Standard_SO("2KJ4_DB", strSelect);
    if ( aResult.length == 1 )
    {
        if(aResult[0].MOTERD == "1")
        {
            // external ground screw
            if ( ECL_ValueSelected("ERDUNGSSCHRAUBE_OPTION_2KJ4","T") )
                addShortDescription( aShortDescriptions, 'AUSSENERD', 'mit' );
        }
    }
        
    // neutral terminal box cover
    if ( ECL_ValueSelected("ANSCHLUSSKASTEN_NEUTRAL_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'KKDEKL', 'Neutral' );

    // cast terminal box cover
    if ( ECL_ValueSelected("ANSCHLUSSKASTEN_VERGOSSEN_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'KKABDICH', 'vergossen' );
        
    // motor connector
    var strConnector = ECL_GetValue("MOTORSTECKER_2KJ4");
    if(strConnector == "M" || strConnector == "NPT")
    {	
        // Kabelverschraubung
        addShortDescription( aShortDescriptions, 'KKVS', strConnector );
    }
    else
    {
        addShortDescription( aShortDescriptions, 'KKMAT', strConnector );
        
        // mating gets a separate short description, but only if a connector is selected
        var strMating = ECL_GetValue("GEGENSTECKER_2KJ4");
        if ( strMating !== 'ohne' )
        {
            addShortDescription( aShortDescriptions, 'KKMAT_ZUB', strMating );
        }
    }
}


/**
 * @constructor
 * add short descriptions for the other motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for other motor options descriptions
 */
function _addDescriptionsForOtherMotorOptions( oGear, aShortDescriptions )
{
    // motor backstop
    if ( ECL_ValueSelected("RUECKLAUFSPERRE_MOTOR_OPTION_2KJ4","T") )
    {
        addShortDescription( aShortDescriptions, 'MOTRLSP', 'X' );
    }
    
    // motor second shaft end
    if ( ECL_ValueSelected("ZWEITES_WELLENENDE_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'MOZWE', 'mit' );
    
    // handwheel
    if ( ECL_ValueSelected("HANDRAD_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'MOTHANDRAD', 'D' );
    
    // protection cover
    if ( ECL_ValueSelected("SCHUTZDACH_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'BELSDACH',  "W" );
    
    // condensarion drain hole
    if ( ECL_ValueSelected("KONDENSWASSERBOHRUNG_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'MOKOWALO', 'mit' );
    
    // inside corrosion protection
    if ( ECL_ValueSelected("INNENKORROSIONSSCHUTZ_OPTION_2KJ4","T") )
        addShortDescription( aShortDescriptions, 'MOINNENKORSCH', 'ja' );
}

function _addAdapterKADescriptions( oGear, aShortDescriptions )
{
    var strMLFB_AVTYP = "MLFB_AVTYP";
    
    var strSelect = "SELECT DISTINCT MLFB_2KJ_12_KA FROM "+strMLFB_AVTYP+" WHERE ";
    strSelect = strSelect + "AVMLFBTYP = '"+oGear.oRow.MOTTYP+"' AND ";
    strSelect = strSelect + "AVTYPDIM = '"+oGear.oRow.MOTDIM+"' AND ";
    strSelect = strSelect + "LKB = '-' ";
    
    var strResultTable = 'MLFB_ADAPTER_KA';
    if ( dbselect( '2KJ4_DB', strSelect, strResultTable ) == 1 )
    {
        var strKA = dbgetstring( strResultTable, 'MLFB_2KJ_12_KA' );
        if(strKA != "-")
            aShortDescriptions.push("+"+strKA);
    }
}

function _addAdapterKUOptionDescriptions( oGear, aShortDescriptions )
{
    if (ECL_ValueSelected("ANTRIEBSGRUPPE_TYP_2KJ4", "KU")) {
            if (ECL_GetValue("ANTRIEBSGRUPPE_LAGERUNG_2KJ4") == "Zylinderrollenlager") {
                addShortDescription( aShortDescriptions, 'AVLAGER_LGAS', 'Zylinderrollenlager');
            }

            if (ECL_GetValue("ANTRIEBSGRUPPE_DICHTUNG_2KJ4") == "FKM") {
                 addShortDescription( aShortDescriptions, 'DRWERK_AV_LGAS', 'FKM');
            }
            else if (ECL_GetValue("ANTRIEBSGRUPPE_DICHTUNG_2KJ4") == "MSS1-HS") {
                addShortDescription( aShortDescriptions, 'DRWERK_AV_LGAS', 'MSS1-HS');
            }
        }

}

function _addAdapterOptionDescriptions( oGear, aShortDescriptions )
{
    if ( ECL_ValueSelected("RUECKLAUFSPERRE_ADAPTER_OPTION_2KJ4","T") )
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'AVRLSP', 'X' );
    }
    
    if(ECL_ValueSelected("RUTSCHKUPPLUNG_ADAPTER_OPTION_2KJ4", "T"))
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'KE', 'Rutschkupplung' );
        
        //eingestelltes Rutschmoment
        var dTorque=ECL_GetValue("RUTSCHKUPPLUNG_MOMENT_2KJ4");
        aShortDescriptions.push( "+Y00'*RKD@" + dTorque + "*'" );
    }
    
    /*
    if ( oGear.oConfig.bDriveGroupSpeedMonitor )
    {
        addShortDescription( aShortDescriptions, 'DRZAHLWACH', 'mit' );
    }
    */
    
    //Kondenswasserbohrung
    if ( ECL_ValueSelected("KONDENSWASSERLOCH_ADAPTER_OPTION_2KJ4","T") )
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'AVKOWALO', 'mit' );
    }
    
    //2-polige Abtriebswelle bei K2 Adapter
    if ( ECL_ValueSelected("ECSI_2POL_MOTORABTRIEBSWELLE_2KJ4","T") )
    {
        addShortDescription( aShortDescriptions, 'MOTBAUART', '2' );
    }
    
    //KQS Antriebsgruppen immer mit glatter Welle, also mit KA A08
    /*
    if(oGear.GearData.MOTTYP == "KQS" || oGear.GearData.MOTTYP == "KFS")
        aShortDescriptions.push('+A08');
        
    */
    
    //Y00 Kurzangaben für Motordrehzahl und Getriebenennmoment
    // Entfernt für SIZER Release 2022.06  // Nur noch für ATEX Adapteranbau
    
    var strATEX_Adapter = ECL_GetValue("ECSI_ATEX_KENNER_2KJ4");
    
    if(ECL_GetValue ("ECSI_2KJ_TYPE_2KJ4") == "GETRIEBE.ADAPTER" && strATEX_Adapter=="ATEX_ADAPTER")
    {
        var strMotorRatedPower = ECL_GetValue("ECSI_CX_MotorRatedPower");
        var strMotorRatedSpeed = ECL_GetValue("ECSI_CX_MotorRatedSpeed");
        
        if( !strMotorRatedPower || !strMotorRatedSpeed)
        {
            //Error
            ECL_SelectValue ("ECSI_STATUS", "E");
            ECL_SetValue ("ECSI_STATUS_MESSAGE", translate("ERRORMSG_MLFB_SIMOGEAR_2KJ___ADAPTER_CONTEXT_ERROR"));
        }
        else
        {	
            aShortDescriptions.push( "+Y00'*ANL@" + strMotorRatedPower + "*'" );
            aShortDescriptions.push( "+Y00'*AND@" + strMotorRatedSpeed + "*'" );
        }
    }
    
    /*
    if(oGear.strProduct == "GETRIEBE.ADAPTER.ATEX" || oMotoxHandler.strProduct == "GETRIEBE.ADAPTER.ATEX_SIP")
    {
        var strmaxAntriebsleistung = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMotorPower, 'kW');
        var strminAntriebsdrehzahl = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMinDrehzahl, '1/min');
        var strmaxAntriebsdrehzahl = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMaxDrehzahl, '1/min');
        
        aShortDescriptions.push( "+Y00'*ANLMX@" + strmaxAntriebsleistung + "*'" );
        aShortDescriptions.push( "+Y00'*ANDMX@" + strmaxAntriebsdrehzahl + "*'" );
        aShortDescriptions.push( "+Y00'*ANDMN@" + strminAntriebsdrehzahl + "*'" );
    }
    */
    
}