//@legacy
//#mode(full-ecma)


//@import "Parser.common.js"


/**
 * @category   SimoGear Parser
 * @package    Parser
 * @copyright  Copyright (c) 2012 Plan Software GmbH (http://www.plansoft.de)
 */


/**
 * decode the encoded short descriptions in the text following OPT(a), add them to the short descriptions of the MLFB
 * and remove all references to OPT(a) from the full MLFB and the full freetext
 *
 * @param strMLFB - normalized MLFB
 * @param strFreetext - freetext input from user (may be empty)
 *
 * @return object: bSuccess - true, if a decoding was performed successfully
 *                            false, if no encoding was performed
 *                 strMLFB - in case the decoding was successful, this value contains the modified MLFB
 *                 strFreetext - in case the decoding was successful, this value contains the modified freetext
 */
function decodeOpt( strMLFB, strFreetext )
{
    var oResult = {
        bSuccess: false,
        strMLFB: strMLFB,
        strFreetext: strFreetext
    };
    
    var oData = parseFreetext( strFreetext, true );
    if ( !( 'Y00' in oData ) )
        return( oResult );
        
    // *OPT(a) must be present in the Y00-freetext (it is not mentioned in the normalized MLFB)
    var strY00Text = oData.Y00;
    var nIndex = strY00Text.indexOf( '*OPT@' );
    if ( nIndex < 0 )
        return( oResult );
        
    // by construction, all text between *OPT(a) and the next * consists of encoded options
    var strEncodedOptions = '', strRemainingText = strY00Text.substring( 0, nIndex );
    var nStartRemaining = -1;
    for( var i = nIndex + 7; i < strY00Text.length; ++i )
    {
        var c = strY00Text.charAt( i );
        if ( c === '*' )
        {
            nStartRemaining = i;
            break;
        }
        strEncodedOptions += c;
    }
    if ( ( nStartRemaining > -1 ) && ( nStartRemaining < (strY00Text.length-1) ) )	// if the closing '*' is the last character, it must not be added
        strRemainingText += strY00Text.substring( nStartRemaining + 1 );
    if ( strEncodedOptions.charAt( 0 ) !== '+' )
        strEncodedOptions = '+' + strEncodedOptions;
        
    // adjust free text and mlfb
    oResult.bSuccess = true;
    oResult.strFreetext = '';
    for ( var opt in oData )
    {
        if ( opt === 'Y00' )
        {
            if ( strRemainingText !== '' )
                oResult.strFreetext += '{Y00:' + strRemainingText + '}';
        }
        else if ( opt !== 'strError' )
            oResult.strFreetext += '{' + opt + ':' + oData[opt] + '}';
    }
    if ( strRemainingText === '' )
        oResult.strMLFB = replace( strMLFB, '+Y00', '' ) + strEncodedOptions;
    else oResult.strMLFB = strMLFB + strEncodedOptions;
        
    return( oResult );
}


/**
 * decode the encoded values from an Y00 short description freetext
 *
 * @param strY00Text - freetext of Y00 short description
 *
 * @return object: may be null on empty text or when no attribute is handled
 *                 bDocumentationCustomerNumberPresent - true if customer number for documentation is present
 *                            							 false, if customer number for documentation is not present
 *                 strDocumentationCustomerNumber - documentation customer number (only valid if bDocumentationCustomerNumberPresent is true, may be empty even when valid)
 *                 bDocumentationFreetextPresent - true if free text for documentation is present
 *                 strDocumentationFreetext - documentation free text (only valid if bDocumentationFreetextPresent is true, may be empty even when valid)
 */
function decodeY00( strY00Text )
{
    if ( strY00Text === '' )
        return( {} );
    
    // extract an option from the Y00 freetext
    var _extractOpt = function( strOpt )
    {
        // normal for of an encoded option: *opt(a)....* or *opt@....*
        var nIndex = strY00Text.indexOf( strOpt );
        if ( nIndex < 0 )
            return( -1 );
        
        var strValue = replace( strY00Text.substring( nIndex ), strOpt, '' );
        nIndex = strValue.indexOf( '*' );
        if ( nIndex > 0 )
            strValue = strValue.substring( 0, nIndex );
        
        return( strValue );
    }
    
    var oText = 
    {
        bDocumentationCustomerNumberPresent: true,
        strDocumentationCustomerNumber: '',
        bDocumentationFreetextPresent: true,
        strDocumentationFreetext: '',
        bAdapterRutschkupplung_DrehmomenteingestelltPresent: true,
        strAdapterRutschkupplung_Drehmomenteingestellt: '',
        bAdapterAntriebsdrehzahlPresent: true,
        strAdapterAntriebsdrehzahl: '',
        bAdapterAntriebsleistungPresent: true,
        strAdapterAntriebsleistung: '',
        bAdapterMaxAntriebsleistungATEXPresent: true,
        strAdapterMaxAntriebsleistungATEX: '',
        bAdapterMaxAntriebsDrehzahlATEXPresent: true,
        strAdapterMaxAntriebsDrehzahlATEX: '',
        bAdapterMinAntriebsDrehzahlATEXPresent: true,
        strAdapterMinAntriebsDrehzahlATEX: ''
    };

    // documentation customer number
    var value = _extractOpt( '*LSK@' );
    if ( value === -1 ){
        oText.bDocumentationCustomerNumberPresent = false;
    }
    else{ 
        oText.strDocumentationCustomerNumber = value;
    }
    // documentation free text
    value = _extractOpt( '*LSF@' );
    if ( value === -1 ){
        oText.bDocumentationFreetextPresent = false;
    }
    else{
         oText.strDocumentationFreetext = value;
    }
    //Rutschkupplung Drehmoment
    value = _extractOpt( '*RKD@' );
    if ( value === -1 ){
        oText.bAdapterRutschkupplung_DrehmomenteingestelltPresent = false;
    }
    else{ 
        oText.strAdapterRutschkupplung_Drehmomenteingestellt = value;
    }
    // Antriebsdrehzahl
    value = _extractOpt( '*AND@' );
    if ( value === -1 ){
        oText.bAdapterAntriebsdrehzahlPresent = false;
    }
    else{ 
        oText.strAdapterAntriebsdrehzahl = value;
    }

    // Antriebsleistung
    value = _extractOpt( '*ANL@' );
    if ( value === -1 ){
        oText.bAdapterAntriebsleistungPresent = false;
    }
    else{
         oText.strAdapterAntriebsleistung = value;
    }
    //ATEX Zusatzdaten
    // MAX Antriebsleistung
    value = _extractOpt( '*ANLMX@' );
    if ( value === -1 ){
        oText.bAdapterMaxAntriebsleistungATEXPresent = false;
    }
    else{
         oText.strAdapterMaxAntriebsleistungATEX = value;
    }
    // MAX Antriebsdrehzahl
    value = _extractOpt( '*ANDMX@' );
    if ( value === -1 ){
        oText.bAdapterMaxAntriebsDrehzahlATEXPresent = false;
    }
   else{
     oText.strAdapterMaxAntriebsDrehzahlATEX = value;
   }

    // MIN Antriebsdrehzahl
    value = _extractOpt( '*ANDMN@' );
    if ( value === -1 ){
        oText.bAdapterMinAntriebsDrehzahlATEXPresent = false;
    }
    else{
         oText.strAdapterMinAntriebsDrehzahlATEX = value;
    }
    
    if ( oText.bDocumentationCustomerNumberPresent || oText.bDocumentationFreetextPresent || oText.bAdapterAntriebsleistungPresent || oText.bAdapterRutschkupplung_DrehmomenteingestelltPresent || oText.bAdapterAntriebsdrehzahlPresent || oText.bAdapterMaxAntriebsleistungATEXPresent || oText.bAdapterMaxAntriebsDrehzahlATEXPresent || oText.bAdapterMinAntriebsDrehzahlATEXPresent )	
        return( oText );

    return( {} );//leeres Object zurück geben
}
