//@legacy
//#mode(full-ecma)

/**
 * @category   MOTOX MLFB generation
 * @package    product logic
 * @copyright  Copyright (c) 2012 Plan Software GmbH (http://www.plansoft.de)
 */


/**
 * @constructor
 * get short descriptions for the motor attributes
 * 
 * @param oGear - geared motor object
 *
 * @return array - short descriptions for the motor attributes,
 *                 each index contains a string with one short description (may be empty)
 */
function getMotorDescriptions( oGear )
{
    var aShortDescriptions = [];
    
    _addKKLageDescriptions( oGear, aShortDescriptions );
    _addElectricDescriptions( oGear, aShortDescriptions );
    _addMechanicalDescriptions( oGear, aShortDescriptions );
    _addDescriptionsForOtherMotorOptions( oGear, aShortDescriptions );
    _addG115MShort_Descriptions( oGear, aShortDescriptions );

    return( aShortDescriptions );
}

function _addKKLageDescriptions( oGear, aShortDescriptions )
{
    var strKKLage=ECL_GetValue("KLEMMKASTENLAGE_2KJ8");
    var strstrMLFB_8=ECL_GetValue("ECSI_MOTORSPANNUNG_SPG_MLFB_2KJ8");
    var strG115_Type = ECL_GetValue("ECSI_G115_TYPE_2KJ8");
    
    
    //zuerst KA für G115D bestimmen
    var strG115D_SD="";
    var strG115M_POS_EXTANB="";
    var strG115_Type = ECL_GetValue("ECSI_G115_TYPE_2KJ8");
    var strKKMAT=ECL_GetValue("MOTORSTECKER_2KJ8");
    var strKKMAT_ZUB=ECL_GetValue("GEGENSTECKER_2KJ8");
    var strSCHUTZART=ECL_GetValue("SCHUTZART_2KJ8");
    
    var strSelect="SELECT DISTINCT G115M_POS_EXTANB, G115_ADPT_OPTION FROM G115_ADPT WHERE "
    strSelect=strSelect+"MOTDIM='"+oGear.oRow.MOTDIM+"' AND ";
    strSelect=strSelect+"KKMAT='"+strKKMAT+"' AND ";
    strSelect=strSelect+"KKMAT_ZUB='"+strKKMAT_ZUB+"' AND ";
    
    if(strG115_Type=="MM")
        strSelect = strSelect + "G115M = '1' AND ";
    else if(strG115_Type=="WM")
        strSelect = strSelect + "G115D = '1' AND ";
    
    strSelect=strSelect+"G115_BUSSYSTEM='"+ECL_GetValue("G115_KOMMUNIKATION_2KJ8")+"' AND ";
    strSelect=strSelect+"G115_BUSSYSTEM_VS='"+replace(ECL_GetValue("G115_ANSCHLUSS_2KJ8"),'_Zoll','"')+"' AND ";
    strSelect=strSelect+"G115_BUSSYSTEM_POS='"+ECL_GetValue("G115_POSITION_KOMM_2KJ8")+"' AND ";
    strSelect=strSelect+"G115_EXTANB='"+ECL_GetValue("G115_VERSORGUNG_2KJ8")+"' AND ";

     // V71 interne Versorgung
     if(ECL_ValueSelected("G115_VERSORGUNG_INTERN_2KJ8", "T"))
     {
         strSelect=strSelect+"G115_INTERN_24V='24VDC' AND ";
     }
     else
     {
         strSelect=strSelect+"G115_INTERN_24V='-' AND ";
     }

    strSelect=strSelect+"SCHUTZART='"+strSCHUTZART+"' AND ";
    strSelect=strSelect+"REGION like'S%' AND ";
    strSelect=strSelect+"AKTIV='1' ";
        
    var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);

    if(aResult.length==1)
    {
        strG115D_SD=aResult[0].G115_ADPT_OPTION;
        strG115M_POS_EXTANB=aResult[0].G115M_POS_EXTANB;
    }
    
    
    //Jetzt KA für Klemmkastenlage
    if(strG115M_POS_EXTANB!="-")
        strKKLage=strKKLage+strG115M_POS_EXTANB;
    
    var strSelect="SELECT DISTINCT MLFB_OPTION FROM MLFB8_KKLAGE_MLFB, MLFB8_G115_KKLAGE WHERE ";
    strSelect=strSelect+"MLFB8_G115_KKLAGE.KKLAGE='"+strKKLage+"' AND ";
    
    strSelect=strSelect+"MLFB_2KJ8_13='"+strstrMLFB_8+"' AND ";
    strSelect=strSelect+"G115_ADPT_OPTION='"+strG115D_SD+"' AND ";
    strSelect=strSelect+"REGION='SM' AND ";
    
    if(strG115_Type=="MM")
    {
        strSelect = strSelect + "MLFB8_KKLAGE_MLFB.G115 = 'M' AND ";
        strSelect = strSelect + "MLFB8_G115_KKLAGE.G115 = 'M' AND ";
    }
    else if(strG115_Type=="WM")
    {
        strSelect = strSelect + "MLFB8_KKLAGE_MLFB.G115 = 'W' AND ";
        strSelect = strSelect + "MLFB8_G115_KKLAGE.G115 = 'W' AND ";
    }
    
    strSelect = strSelect + "MLFB8_KKLAGE_MLFB.KKLAGE_MLFB=MLFB8_G115_KKLAGE.KKLAGE_MLFB ";
        
    var aResult = Object.fromDatabase("2KJ8_DB", strSelect);
    if(aResult.length ==1)
    {
        aShortDescriptions.push( '+' + aResult[0].MLFB_OPTION);
    }
}

function getAdapterDescriptions( oGear )
{
    var aShortDescriptions = [];
    
    _addAdapterKADescriptions( oGear, aShortDescriptions );
    
    _addAdapterOptionDescriptions( oGear, aShortDescriptions );
    
    
    //_addATEXDescriptions( oGear, aShortDescriptions );
    
    return( aShortDescriptions );
}


/**
 * @constructor
 * add short descriptions for the motor ATEX options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for motor ATEX options descriptions
 */
function _addATEXDescriptions( oGear, aShortDescriptions )
{
    //ATEX Getriebekennzeichnung
    var strSelect = "";
    var strDBName = "SIMOGEAR_DB"; 
    var strDBResult = "ATEX_GETRIEBE_KENNZEICHNUNG_MLFB";
    var strGetriebeKennzeichnungATEX;
    
    strSelect = "SELECT ATEXKNZGETR FROM ATEXKNZGETR WHERE ";
    strSelect += "ZONE = '"+oGear.oAtex.strAtexZone+"' AND ATEXKATEGORIE = '"+oGear.oAtex.strAtexExplosionGroup+"'  AND EGGEHOLK = '"+oGear.oConfig.strOilLevelControl+"' ";
    
    if ( dbselect( strDBName, strSelect, strDBResult ) > 0 )
    {
        strGetriebeKennzeichnungATEX = dbgetstring(strDBResult,'ATEXKNZGETR');
    }
    else
    {
        logtraceln("Fehler bei Bestimmung der ATEX Kennzeichnung des Getriebes! SQL="+strSelect);
    }
    
    addShortDescription( aShortDescriptions, 'ATEXKNZGETR', strGetriebeKennzeichnungATEX );
    
    //ATEX Zone
    var strATEXZone = oGear.oAtex.strAtexZone;
    addShortDescription( aShortDescriptions, 'ATEXZONE', strATEXZone );
    
    //ATEX Temperautklasse T4
    var bAtexClassT4 = oGear.oAtex.bAtexClassT4;
    if(bAtexClassT4)
        addShortDescription( aShortDescriptions, 'ATEXTK', 'T4' );
        
    //ATEX Oberflächentemperatur
    var bAtexSurfaceTemp = oGear.oAtex.bAtexSurfaceTemp;
    if(bAtexSurfaceTemp)
        addShortDescription( aShortDescriptions, 'ATEXTK', '120°C' );
        
    //ATEX maximale Umgebungstemperatur
    var strAtexUmgebTemp = oGear.oAtex.strAtexTemperature+"°C";
        addShortDescription( aShortDescriptions, 'ATEXUMTEMP',  strAtexUmgebTemp);
}


/**
 * @constructor
 * add short descriptions for the electric motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for electric motor options descriptions
 */
function _addElectricDescriptions( oGear, aShortDescriptions )
{
    // motor efficiency class is already encoded in the MLFB trunk (digit 12)
    
    // varying motor power is computed with the trunk MLFB-part for the motor
    
    // poles
    addShortDescription( aShortDescriptions, 'POLZAHL', oGear.oRow.POLZAHL );
    
    // electrical specification is computed with the trunk MLFB-part for the motor
    
    // motor voltage is encoded in the trunk MLFB-part for the motor (and in its necessary short description)

    // thermical motorprotection
    var strProtection = ECL_GetValue("MOTORSCHUTZ_2KJ8");
    if ( strProtection !== 'ohne' )
        addShortDescription( aShortDescriptions, 'WIELART', strProtection );		
    
    // anti-condensation heating
    var strHeating = ECL_GetValue("STILLSTANDSHEIZUNG_2KJ8");
    if ( strHeating !== 'ohne' )
        addShortDescription( aShortDescriptions, 'WIELHEIZ', 'mit', 'WIELHEIZSPG', strHeating );		

    // winding temperature class
    var strClass = ECL_GetValue("WAERMEKLASSE_2KJ8");
    if ( strClass !== 'F' )
        addShortDescription( aShortDescriptions, 'MOTWK', strClass );		
    
    // reinforced insulation
    /*
    if ( oGear.oConfig.bReinforcedInsulation )
        MLFB.MOTOX.SD.addShortDescription( aShortDescriptions, 'VERSTISO', 'ja' );
    */
    
    // increased protection for special environments
    if ( ECL_ValueSelected("SAEURESCHUTZ_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'MOFSSCHUTZWI', 'ja' );
}


/**
 * @constructor
 * add short descriptions for the mechanical motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for mechanical motor options descriptions
 */
function _addMechanicalDescriptions( oGear, aShortDescriptions )
{
    // motor protection
    addShortDescription( aShortDescriptions, 'SCHUTZART', ECL_GetValue("SCHUTZART_2KJ8") );

    // mechanical venting
    var strVenting = ECL_GetValue("BELUEFTUNG_2KJ8");
    if ( strVenting !== 'Standard' )
    {
        if ( strVenting === 'Schwungradluefter' )
            addShortDescription( aShortDescriptions, 'MOTSWRD', 'I', 'BELLUEFT', 'Schwungradluefter' );		
        else if ( strVenting === 'Zugeordneter Luefter' )
        {
            addShortDescription( aShortDescriptions, 'BELART', 'F' );
            
            // there are no additional short descriptions for protection and voltage of 'venting with foreign blower'
        }
        else 
            addShortDescription( aShortDescriptions, 'MOTSWRD', '-', 'BELLUEFT', strVenting );	
    }
    
    // external ground screw
    //wird unterdrück im 2KJ8
    //if ( ECL_ValueSelected("ERDUNGSSCHRAUBE_OPTION_2KJ8","T") )
    //	addShortDescription( aShortDescriptions, 'AUSSENERD', 'mit' );
        
    // neutral terminal box cover
    if ( ECL_ValueSelected("ANSCHLUSSKASTEN_NEUTRAL_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'KKDEKL', 'Neutral' );

    // cast terminal box cover
    if ( ECL_ValueSelected("ANSCHLUSSKASTEN_VERGOSSEN_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'KKABDICH', 'vergossen' );
        
    // motor connector
    var strG115_Type = ECL_GetValue("ECSI_G115_TYPE_2KJ8");
    var strConnector = ECL_GetValue("MOTORSTECKER_2KJ8");
    
    if(strConnector!= "" && strG115_Type=="WM")
    {
        addShortDescription( aShortDescriptions, 'KKMAT', strConnector );
        
        // mating gets a separate short description, but only if a connector is selected
        var strMating = ECL_GetValue("GEGENSTECKER_2KJ8");
        if ( strMating !== 'ohne' )
        {
            addShortDescription( aShortDescriptions, 'KKMAT_ZUB', strMating );
        }
    }
}


/**
 * @constructor
 * add short descriptions for the other motor options to the array of short descriptions
 * 
 * @param oGear - geared motor object
 * @param aShortDescriptions - array with short descriptions that should be enhanced for other motor options descriptions
 */
function _addDescriptionsForOtherMotorOptions( oGear, aShortDescriptions )
{
    // motor backstop
    if ( ECL_ValueSelected("RUECKLAUFSPERRE_MOTOR_OPTION_2KJ8","T") )
    {
        addShortDescription( aShortDescriptions, 'MOTRLSP', 'X' );
    }
    
    // motor second shaft end
    if ( ECL_ValueSelected("ZWEITES_WELLENENDE_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'MOZWE', 'mit' );
    
    // handwheel
    if ( ECL_ValueSelected("HANDRAD_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'MOTHANDRAD', 'D' );
    
    // protection cover
    if ( ECL_ValueSelected("SCHUTZDACH_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'BELSDACH',  "W" );
    
    // condensarion drain hole
    if ( ECL_ValueSelected("KONDENSWASSERBOHRUNG_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'MOKOWALO', 'mit' );
    
    // inside corrosion protection
    if ( ECL_ValueSelected("INNENKORROSIONSSCHUTZ_OPTION_2KJ8","T") )
        addShortDescription( aShortDescriptions, 'MOINNENKORSCH', 'ja' );
}

function _addAdapterKADescriptions( oGear, aShortDescriptions )
{
    var strMLFB_AVTYP = "MLFB_AVTYP";
    
    var strSelect = "SELECT DISTINCT MLFB_2KJ_12_KA FROM "+strMLFB_AVTYP+" WHERE ";
    strSelect = strSelect + "AVMLFBTYP = '"+oGear.oRow.MOTTYP+"' AND ";
    strSelect = strSelect + "AVTYPDIM = '"+oGear.oRow.MOTDIM+"' AND ";
    strSelect = strSelect + "LKB = '-' ";
    
    var strResultTable = 'MLFB_ADAPTER_KA';
    if ( dbselect( '2KJ8_DB', strSelect, strResultTable ) == 1 )
    {
        var strKA = dbgetstring( strResultTable, 'MLFB_2KJ_12_KA' );
        if(strKA != "-")
            aShortDescriptions.push("+"+strKA);
    }
}

function _addAdapterOptionDescriptions( oGear, aShortDescriptions )
{
    if ( ECL_ValueSelected("RUECKLAUFSPERRE_ADAPTER_OPTION_2KJ8","T") )
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'AVRLSP', 'X' );
    }
    
    if(ECL_ValueSelected("RUTSCHKUPPLUNG_ADAPTER_OPTION_2KJ8", "T"))
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'KE', 'Rutschkupplung' );
        
        //eingestelltes Rutschmoment
        var dTorque=ECL_GetValue("RUTSCHKUPPLUNG_MOMENT_2KJ8");
        aShortDescriptions.push( "+Y00'*RKD@" + dTorque + "*'" );
    }
    
    /*
    if ( oGear.oConfig.bDriveGroupSpeedMonitor )
    {
        addShortDescription( aShortDescriptions, 'DRZAHLWACH', 'mit' );
    }
    */
    
    //Kondenswasserbohrung
    if ( ECL_ValueSelected("KONDENSWASSERLOCH_ADAPTER_OPTION_2KJ8","T") )
    {
        addShortDescription( aShortDescriptions, 'AVMLFBTYP', '', 'AVKOWALO', 'mit' );
    }
    
    //2-polige Abtriebswelle bei K2 Adapter
    if ( ECL_ValueSelected("ECSI_2POL_MOTORABTRIEBSWELLE_2KJ8","T") )
    {
        addShortDescription( aShortDescriptions, 'MOTBAUART', '2' );
    }
    
    //KQS Antriebsgruppen immer mit glatter Welle, also mit KA A08
    /*
    if(oGear.GearData.MOTTYP == "KQS" || oGear.GearData.MOTTYP == "KFS")
        aShortDescriptions.push('+A08');
        
    */
    
    //Y00 Kurzangaben für Motordrehzahl und Getriebenennmoment
    // Entfernt für SIZER Release 2022.06
    /*
    if(ECL_GetValue ("ECSI_2KJ_TYPE_2KJ8") == "GETRIEBE.ADAPTER")
    {
        var strMotorRatedPower = ECL_GetValue("ECSI_CX_MotorRatedPower");
        var strMotorRatedSpeed = ECL_GetValue("ECSI_CX_MotorRatedSpeed");
        
        if( !strMotorRatedPower || !strMotorRatedSpeed)
        {
            //Error
            ECL_SelectValue ("ECSI_STATUS", "E");
            ECL_SetValue ("ECSI_STATUS_MESSAGE", translate("ERRORMSG_MLFB_SIMOGEAR_2KJ___ADAPTER_CONTEXT_ERROR"));
        }
        else
        {	
            aShortDescriptions.push( "+Y00'*ANL@" + strMotorRatedPower + "*'" );
            aShortDescriptions.push( "+Y00'*AND@" + strMotorRatedSpeed + "*'" );
        }
    }
    */
    
    /*
    if(oGear.strProduct == "GETRIEBE.ADAPTER.ATEX" || oMotoxHandler.strProduct == "GETRIEBE.ADAPTER.ATEX_SIP")
    {
        var strmaxAntriebsleistung = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMotorPower, 'kW');
        var strminAntriebsdrehzahl = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMinDrehzahl, '1/min');
        var strmaxAntriebsdrehzahl = Utils.TechValue.getValueAsNumber(oGear.oAtex.oMaxDrehzahl, '1/min');
        
        aShortDescriptions.push( "+Y00'*ANLMX@" + strmaxAntriebsleistung + "*'" );
        aShortDescriptions.push( "+Y00'*ANDMX@" + strmaxAntriebsdrehzahl + "*'" );
        aShortDescriptions.push( "+Y00'*ANDMN@" + strminAntriebsdrehzahl + "*'" );
    }
    */
    
}

function _addG115MShort_Descriptions( oGear, aShortDescriptions )
{
    _G115D_MainSD( oGear, aShortDescriptions );
    _G115D_ext24VDCAnbindungSD( oGear, aShortDescriptions );
    _G115D_intern24VDCAnbindungSD( oGear, aShortDescriptions );
    _G115D_SDKarteSD( oGear, aShortDescriptions );
}

function _G115D_MainSD( oGear, aShortDescriptions )
{
    var strG115D_SD="";
    var strG115_Type = ECL_GetValue("ECSI_G115_TYPE_2KJ8");
    var strKKMAT=ECL_GetValue("MOTORSTECKER_2KJ8");
    var strKKMAT_ZUB=ECL_GetValue("GEGENSTECKER_2KJ8");
    var strSCHUTZART=ECL_GetValue("SCHUTZART_2KJ8");
    
    var strSelect="SELECT DISTINCT G115_ADPT_OPTION FROM G115_ADPT WHERE "
    strSelect=strSelect+"MOTDIM='"+oGear.oRow.MOTDIM+"' AND ";
    strSelect=strSelect+"KKMAT='"+strKKMAT+"' AND ";
    strSelect=strSelect+"KKMAT_ZUB='"+strKKMAT_ZUB+"' AND ";
    
    if(strG115_Type=="MM")
        strSelect = strSelect + "G115M = '1' AND ";
    else if(strG115_Type=="WM")
        strSelect = strSelect + "G115D = '1' AND ";
    
    strSelect=strSelect+"G115_BUSSYSTEM='"+ECL_GetValue("G115_KOMMUNIKATION_2KJ8")+"' AND ";
    strSelect=strSelect+"G115_BUSSYSTEM_VS='"+replace(ECL_GetValue("G115_ANSCHLUSS_2KJ8"),'_Zoll','"')+"' AND ";
    strSelect=strSelect+"G115_BUSSYSTEM_POS='"+ECL_GetValue("G115_POSITION_KOMM_2KJ8")+"' AND ";
    strSelect=strSelect+"G115_EXTANB='"+ECL_GetValue("G115_VERSORGUNG_2KJ8")+"' AND ";

     // V71 interne Versorgung
    if(ECL_ValueSelected("G115_VERSORGUNG_INTERN_2KJ8", "T"))
    {
        strSelect=strSelect+"G115_INTERN_24V='24VDC' AND ";
    }
    else
    {
        strSelect=strSelect+"G115_INTERN_24V='-' AND ";
    }

    strSelect=strSelect+"SCHUTZART='"+strSCHUTZART+"' AND ";
    strSelect=strSelect+"REGION like'S%' AND ";
    strSelect=strSelect+"AKTIV='1' ";
        
    var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);
    
    if(aResult.length==1)
    {
        strG115D_SD=aResult[0].G115_ADPT_OPTION;
        if(strG115D_SD!="-")
            aShortDescriptions.push( '+'+strG115D_SD);
    }
    else
        logtraceln("Fehler bei Bestimmung KA G115D!, SQL="+strSelect);
}

function _G115D_ext24VDCAnbindungSD( oGear, aShortDescriptions )
{
    var str24VDC_Anschluss_SD="";
    var str24VDC_Anschluss = ECL_GetValue("G115_VERSORGUNG_2KJ8");
    
    if(str24VDC_Anschluss != "ohne")
        addShortDescription( aShortDescriptions, 'G115_EXTANB', str24VDC_Anschluss );
}

function _G115D_intern24VDCAnbindungSD( oGear, aShortDescriptions )
{
    if(ECL_ValueSelected("G115_VERSORGUNG_INTERN_2KJ8", "T"))
        addShortDescription( aShortDescriptions, 'G115_INTERN_24V', "24VDC");
}

function _G115D_SDKarteSD( oGear, aShortDescriptions )
{
    var strG115_Type = ECL_GetValue("ECSI_G115_TYPE_2KJ8");
    
    if(strG115_Type=="MM")
    {
        if ( ECL_ValueSelected("G115_SPEICHERKARTE_2KJ8","leer") )
            addShortDescription( aShortDescriptions, 'G115_SD_KARTE', 'mit', 'G115_SD_KARTEPARA', 'ohne' );
        else if ( ECL_ValueSelected("G115_SPEICHERKARTE_2KJ8","firmware") )
            addShortDescription( aShortDescriptions, 'G115_SD_KARTE', 'mit', 'G115_SD_KARTEPARA', 'mit' );

        if ( ECL_ValueSelected("G115_FUNCTION_SAFETY_SLS_2KJ8","T") )
            addShortDescription( aShortDescriptions, 'G115_SD_KARTE', 'mit', 'G115_SD_KARTESLS', 'mit' );
    }
}