//@legacy
//#mode(full-ecma)


//@import "../2KJ8_CONFIG/utils.js"

//@import "Parser.common.js"



/**
 * @category   SimoGear Parser
 * @package    Parser
 * @copyright  Copyright (c) 2012 Plan Software GmbH (http://www.plansoft.de)
 */

/**
 * @constructor
 * set the values of all short descriptions to the geared motor or geared motor configuration
 *
 * @param oShortDescriptions - object for short descriptions (and optional corresponding freetext)
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group and freetext (the freetext string may be empty)
 * @param oMotox - geared motor object to adjust
 * @param oParsingData - parsing data object: contains parsing data results from different parsing stages (not all subobjects may be present due to parsing errors)
 *                       oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *									EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                       oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                       oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                       oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                       oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *                  			 MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                               WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, MOTSPG, SUCHTYP etc. (all columns of table CD_MOTOREN)
 */
function _adjustGearedMotorWithShortDescriptions( oShortDescriptions, oMotox, oParsingData, strRegion  )
{
    //--------------------------------------------------------------------------------------------------------
    // process all short descriptions
    //--------------------------------------------------------------------------------------------------------
    var strGroup;
    var strRotatingDirectionPresent = '';
    for ( var strSD in oShortDescriptions )
    {
        //--------------------------------------------------------------------------------------------------------
        // the easy part - the automatic processing
        //--------------------------------------------------------------------------------------------------------
        setShortDescriptionToGear( strSD, oMotox, '2KJ8_KURZANGABEN_DB' );
        
        //--------------------------------------------------------------------------------------------------------
        // the heavy parts: exceptions and additions
        //--------------------------------------------------------------------------------------------------------
        if(oShortDescriptions[strSD])
        {
            strGroup = oShortDescriptions[strSD].group;
            if ( strGroup === 'GETRIEBE_BAUFORM' )
            {
                // the gear output side is also encoded in the short description for the mounting position
                var strSide = strSD.substring( 0, 2 );
                if ( strSide === 'D0' )
                    oMotox.oConfig.ABTRIEBSSEITE = '-';
                else if ( strSide === 'D1' )
                    oMotox.oConfig.ABTRIEBSSEITE = 'A';
                else if ( strSide === 'D2' )
                    oMotox.oConfig.ABTRIEBSSEITE = 'B';
                else if ( strSide === 'D3' )
                    oMotox.oConfig.ABTRIEBSSEITE = 'AB';
                else if ( strSide === 'D4' )
                    oMotox.oConfig.ABTRIEBSSEITE = 'BA';			
                // else bad luck dude - other output sides not defined yet
            }
            else if ( strGroup === 'GETRIEBE_SONDEREINBAULAGE' )
            {
                //Sondereinbaulage 2KJ3308-5LM32-4FF1-Z D04+E16+G23+G60+H09+K01+K08+L02+L75+M55+P90
                var strSE = strSD.left(1);
                var iSE = parseInt(strSD.right(2));
                
                if(strSE == "E")
                {
                    ECL_SetValue("SONDEREINBAULAGE_OPTION_2KJ8", "T");
                    var strWinkel = "";
                    var strAchse = "";
                    
                    var strSelect = "SELECT DISTINCT WERT_1, MERKMAL_1 FROM MLFB8_OPTIONEN WHERE ";
                    strSelect = strSelect + "MLFB_OPTION = '"+strSD+"' AND Aktiv='1'";
                    
                    var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);
                    if ( aResult.length == 1 )
                    {
                        strWinkel = aResult[0].WERT_1;
                        strAchse = aResult[0].MERKMAL_1;
                    }
                    else
                    {
                        logtraceln("Fehler beim Bestimmen des Winkels und der Rotationsachse im Parser bei Sondereinbaulage; SQL="+strSelect);
                        ECL_DeselectAttribute ("SONDEREINBAULAGE_OPTION_2KJ8");
                    }
                    
                    ECL_SetValue("SE_ACHSE_WAHL_2KJ8",strAchse.left(1));
                    ECL_SetValue("SE_ROTATIONSWINKEL_2KJ8",strAchse+"_"+strWinkel);
                    
                    var iMountingPosition = parseInt(ECL_GetValue("ECSI_CX_MountingPosition"));
                    if(isNaN(iMountingPosition))
                    {
                        iMountingPosition = 1;
                    }
                    
                    var strBild;
                    var strGetriebetyp = oMotox.GearData.EGBEZTYP+oMotox.GearData.EGBEZUEBS;
                    strGetriebetyp = replace(strGetriebetyp, "-", "");
                    strGetriebetyp = left(strGetriebetyp,1);

                    var strSelect = "SELECT DISTINCT BILD FROM BILDER_SONDEREINBAULAGEN_NEW WHERE ";
                    strSelect = strSelect + "Rotationsachse = '"+strAchse.left(1)+"' AND ";
                    strSelect = strSelect + "EGPOSHL = 'M"+iMountingPosition+"' AND ";
                    strSelect = strSelect + "Getriebetyp = '"+strGetriebetyp+"' ";
                    
                    var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);
                    if ( aResult.length == 1 )
                    {
                        strBild = aResult[0].BILD;
                    }
                    else
                    {
                        logtraceln("Fehler beim Bestimmen der Zielposition im Parser bei Sondereinbaulage; SQL="+strSelect);
                        ECL_DeselectAttribute ("SONDEREINBAULAGE_OPTION_2KJ8");
                    }
                    
                    ECL_SetValue("SE_ZIELPOSITION_2KJ8",strBild);
                }	
            }
            else if ( strGroup === 'GETRIEBE_DREHRICHTUNG_ABTRIEBSWELLE' )
                strRotatingDirectionPresent = strSD;
            else if ( ( strSD === 'G72' ) || ( strSD === 'N23' ) || ( strSD === 'A15' ) )
                oMotox.oConfig.RUECKLAUFSPERRE_MOTOR_OPTION = "T";
            else if ( strGroup === 'ENVIRONMENT_TEMPERATURE' )
            {
                // if ambiet temperate is <= 40°C then there could be a SD K97 present
                if(strSD=="K97")
                    ECL_SetValue("AMBIENT_TEMPERATURE_2KJ8", "30_40");
                
            }
    /* -og- 120402 mating have its own short descriptions that are independent of the connector			
            // mating of motor connectors depend to the connector itself
            else if ( strSD === 'N18' )
            {
                if ( ( 'N00' in oShortDescriptions ) || ( 'N01' in oShortDescriptions ) )
                    oMotox.oConfig.strMating = 'MAK_HAN10_2';
                else if ( ( 'N04' in oShortDescriptions ) || ( 'N06' in oShortDescriptions ) )
                    oMotox.oConfig.strMating = 'MAK_HAN10_1';
                // else bad luck dude - combination not specified at the time of this writing
            }
            else if ( strSD === 'N19' )
            {
                if ( ( 'N08' in oShortDescriptions ) || ( 'N09' in oShortDescriptions ) )
                    oMotox.oConfig.strMating = 'MAK_K4/4_1';
                else if ( ( 'N10' in oShortDescriptions ) || ( 'N11' in oShortDescriptions ) )
                    oMotox.oConfig.strMating = 'MAK_K4/4_2';
                // else bad luck dude - combination not specified at the time of this writing
            }
    */		
            // the conservation also determines the lacquer type
            else if (left(strSD,1) == "L" && strSD != "L19" && strSD != "L11" && strSD != "L12" && strSD != "L27")
            {
                if( strSD === 'L00' )
                    oMotox.oConfig.OBERFLAECHENBEHANDLUNG = 'UNLACKIERT';
                else if ( ( strSD === 'L01' ) || ( strSD === 'L08' ) || ( strSD === 'L09' ) || ( strSD === 'L10' ) || ( strSD === 'L22' ) || ( strSD === 'L24' ) || ( strSD === 'L26' ) || ( strSD === 'L28' ) )
                    oMotox.oConfig.OBERFLAECHENBEHANDLUNG = 'GRUNDIERT';
                else oMotox.oConfig.OBERFLAECHENBEHANDLUNG = 'LACKIERT';
            }
            else if ( strGroup === 'MOTORSPEZIFIKATION_CHINA' )
            {
                oMotox.ECSI_SPEZIFIKATION_CHECK = "CHINA";
            }
            else if ( strGroup === 'MOTORSPEZIFIKATION_EAC' )
            {
                oMotox.ECSI_SPEZIFIKATION_CHECK = "EAC";
            }
            else if ( strGroup === 'MOTORSPEZIFIKATION_CSA_UL_R' )
            {
                oMotox.ECSI_SPEZIFIKATION_CHECK = "ULR_CSA";
            }
        }
    }
    
    //--------------------------------------------------------------------------------------------------------
    // special handling for backstop
    //--------------------------------------------------------------------------------------------------------
    if ( oMotox.ECSI_2KJ_TYPE=="GETRIEBEMOTOR" )
    {
        if ( strRotatingDirectionPresent != '' )
        {
            ECL_EnableValue("DREHRICHTUNG_MOTOR_OPTION_2KJ8", "T");
            ECL_SelectValue("DREHRICHTUNG_MOTOR_OPTION_2KJ8", "T");
            
            var strEGTYP = oParsingData.oGearType.EGBEZTYP+oParsingData.oGearType.EGBEZUEBS;
            strEGTYP=replace(strEGTYP,"-","");
            
            var strSelect = "SELECT DISTINCT ANTRIEBSDREHRICHTUNG_MOTOR,ABTRIEBSDREHRICHTUNG_GETRIEBE FROM ABTRIEBSDREHRICHTUNGEN WHERE ";
            strSelect = strSelect + "GETRIEBE = '"+strEGTYP+"' AND ";
            
            strSelect = strSelect + "ABTRIEBSSEITE = '"+oMotox.oConfig.ABTRIEBSSEITE+"' AND ";
            
            strSelect = strSelect + "KA_RLSP='"+strRotatingDirectionPresent+"' ";
            
            
            var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);
            if(aResult.length==1)
            {
                var strRichtungMotor = aResult[0].ANTRIEBSDREHRICHTUNG_MOTOR;
                var strRichtungGetriebe = aResult[0].ABTRIEBSDREHRICHTUNG_GETRIEBE;
                
                if(strRichtungMotor == "CW")
                {
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_MOTOR_2KJ8", "rechts");
                }
                else
                {
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_MOTOR_2KJ8", "links");
                }
                
                if(strRichtungGetriebe == "CW")
                {
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "rechts");
                }
                else
                {
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_ABTRIEBSSEITE_2KJ8", "links");
                }
            }
            else
                logtraceln("Fehler Bestimmung Abtriebsdrehrichtung Motor und Getriebe, SQL="+strSelect);
        
        }
        else
        {
            ECL_EnableValue("DREHRICHTUNG_MOTOR_OPTION_2KJ8", "T");
            ECL_DeselectValue("DREHRICHTUNG_MOTOR_OPTION_2KJ8", "T");
        }
    }
    else if ( oMotox.ECSI_2KJ_TYPE=="GETRIEBE.ADAPTER" )
    {
        if ( strRotatingDirectionPresent != '' )
        {
            ECL_EnableValue("DREHRICHTUNG_ADAPTER_OPTION_2KJ8", "T");
            ECL_SelectValue("DREHRICHTUNG_ADAPTER_OPTION_2KJ8", "T");
            
            var strEGTYP = oParsingData.oGearType.EGBEZTYP+oParsingData.oGearType.EGBEZUEBS;
            strEGTYP=replace(strEGTYP,"-","");
            
            var strSelect = "SELECT DISTINCT ANTRIEBSDREHRICHTUNG_MOTOR,ABTRIEBSDREHRICHTUNG_GETRIEBE FROM ABTRIEBSDREHRICHTUNGEN WHERE ";
            strSelect = strSelect + "GETRIEBE = '"+strEGTYP+"' AND ";
            
            strSelect = strSelect + "ABTRIEBSSEITE = '"+oMotox.oConfig.ABTRIEBSSEITE+"' AND ";
            
            strSelect = strSelect + "KA_RLSP='"+strRotatingDirectionPresent+"' ";
            
            
            var aResult = ExecSQL_Standard_SO("2KJ8_DB", strSelect);
            if(aResult.length==1)
            {
                var strRichtungMotor = aResult[0].ANTRIEBSDREHRICHTUNG_MOTOR;
                var strRichtungGetriebe = aResult[0].ABTRIEBSDREHRICHTUNG_GETRIEBE;
                
                if(strRichtungMotor == "CW")
                {
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "rechts");
                }
                else
                {
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_MOTOR_ADAPTER_2KJ8", "links");
                }
                
                if(strRichtungGetriebe == "CW")
                {
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "rechts");
                }
                else
                {
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "links");
                    ECL_ActivateValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "rechts");
                    ECL_SelectValue("DREHRICHTUNG_ABTRIEBSSEITE_ADAPTER_2KJ8", "links");
                }
            }
            else
                logtraceln("Fehler Bestimmung Abtriebsdrehrichtung Motor und Getriebe, SQL="+strSelect);
        
        }
        else
        {
            ECL_EnableValue("DREHRICHTUNG_ADAPTER_OPTION_2KJ8", "T");
            ECL_DeselectValue("DREHRICHTUNG_ADAPTER_OPTION_2KJ8", "T");
        }
    
    }

    //--------------------------------------------------------------------------------------------------------
    // transfer freetext from special short descriptions to their proper location
    //--------------------------------------------------------------------------------------------------------
    if ( 'Y00' in oShortDescriptions )
    {
        // rating plate free text and/or rating plate customer number
        var oTextData = decodeY00( getFreetextValue( oShortDescriptions.Y00.freetext ) );
        if ( oTextData )
        {
            if ( "bDocumentationCustomerNumberPresent" in oTextData ){
                oMotox.oConfig.LEISTUNGSSCHILD_ZUS_KUNDENNR = oTextData.strDocumentationCustomerNumber;
            }
            if ( "bDocumentationFreetextPresent" in oTextData ){
                oMotox.oConfig.LEISTUNGSSCHILD_ZUS_FREITEXT = oTextData.strDocumentationFreetext;
            }
            if ( "bAdapterRutschkupplung_DrehmomenteingestelltPresent" in oTextData ){
                oMotox.oConfig.RUTSCHKUPPLUNG_MOMENT = oTextData.strAdapterRutschkupplung_Drehmomenteingestellt;
            }
        }
    }
    if ( 'Y80' in oShortDescriptions )
    {
        // special coloring
        var strColorID = replace( trimright( trimleft( getFreetextValue( oShortDescriptions.Y80.freetext ), ' ' ), ' ' ), '*ral@', '' );
        var strColorID = replace( trimright( trimleft( strColorID, ' ' ), ' ' ), '*RAL@', '' );
        if ( strColorID !== '' )
            oMotox.oConfig.FARBE = replace( strColorID, '*', '' );
    }
    
    //Setzen der Bauform Attribute
    var strEGPOS_BAUFORM=oMotox.oConfig.BAUFORM;
    if(strEGPOS_BAUFORM)
    {
        if(oMotox.oConfig.ABTRIEBSSEITE != "-")
            strEGPOS_BAUFORM=strEGPOS_BAUFORM+"-"+oMotox.oConfig.ABTRIEBSSEITE;
        
        oMotox.oConfig.ECSI_BAUFORM_EGPOS=strEGPOS_BAUFORM;
    }
}


/**
 * @constructor
 * adjust technical data of a partially constructed geared motor object
 *
 * @param oMotox - geared motor object whose technical data should be adjusted
 * @param oParsingData - results from the parsing process
 *                       oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *								    EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                       oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                       oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                       oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                       oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *         					     MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                               WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, SUCHTYP etc. (all columns of table CD_MOTOREN)
 */
function _adjustTechnicalData( oMotox, oParsingData, strRegion )
{
    var strDBResult = "MOTFHT_DATA";
    // gear data from parsing
    var dDBRatio = oParsingData.oRatio.EGUEB;
    var dDBNominalTorque = oParsingData.oRatio.T2FB1;
    var MMSystem = "SI";
    
    // technical data to calculate
    var dDBInputSpeed,dDBOutputSpeed,dDBPower,dDBOutputTorque;
    
    var BDRHZAHL = oParsingData.oMotor.BDRHZAHL_1;
    
    var LEISTUNG = oParsingData.oMotor.LEISTUNG1;

    var dDBPower = parseFloat(LEISTUNG);
    var strLeistung = replace(LEISTUNG,".", ",");
    
    // calculate technical data
    dDBInputSpeed = BDRHZAHL;
    dDBOutputSpeed = dDBInputSpeed / dDBRatio;

    var nStufenFaktor    = 1.0;
    var nVerlustLeistung = 0;
    var etaPab = 1.0;
    
    //Bei Schneckenstirnradgetrieben wird Wirkungsgrad gesondert berechnet
    if(oMotox.GearData.EGBEZTYP == "C")
    {
        etaPab = getWirkungsgradSchneckengetriebe( oMotox.GearData.EGBEZTYP, oMotox.GearData.EGBEZBRA, oMotox.GearData.EGBEZUEBS, oMotox.GearData.EGBEZEI, oMotox.GearData.EGBEZDIM, "-", "-", "-", dDBRatio, dDBInputSpeed, MMSystem);
    }
    
    var pAB = (dDBPower - nVerlustLeistung) * nStufenFaktor * etaPab;
    
    dDBOutputTorque = ( 9550.0 * pAB ) / dDBOutputSpeed;
    
    //Bei Schneckenstirnradgetrieben wird das Nennmoment gesondert berechnet
    if(oMotox.GearData.EGBEZTYP == "C")
    {
        dDBNominalTorque = getNennmomentSchneckengetriebe( -1, oMotox.GearData.EGBEZTYP, oMotox.GearData.EGBEZBRA, oMotox.GearData.EGBEZUEBS, oMotox.GearData.EGBEZEI, oMotox.GearData.EGBEZDIM, "-", "-", "-", dDBRatio, dDBInputSpeed, MMSystem);
    }
    
    // actual geared motor technical data

    oMotox.GearData.LEISTUNG =  dDBPower;
    
    var strNennmoment_Ausgabe;
    strNennmoment_Ausgabe = RoundMotox( dDBNominalTorque,MMSystem,"DREHMOMENT" );
    
    oMotox.GearData.BDRHZAHL_MOTOR = RoundMotox( dDBInputSpeed, MMSystem, 'DREHZAHL' );
    oMotox.GearData.ABTRIEBSDREHZAHL = RoundMotox( dDBOutputSpeed, MMSystem, 'DREHZAHL' );
        
    oMotox.GearData.NENNMOMENT = parseFloat(dDBNominalTorque);
    oMotox.GearData.NENNMOMENT_2 = parseFloat(dDBNominalTorque);
    oMotox.GearData.UEBERSETZUNG = dDBRatio;
    oMotox.GearData.ABTRIEBSDREHMOMENT = dDBOutputTorque;
    oMotox.GearData.BETRIEBSFAKTOR = dDBNominalTorque / dDBOutputTorque;

    // motor input power
    oMotox.strMotorPower = '' + dDBPower;	
    oMotox.dRatio = dDBRatio;
    oMotox.strInputSpeedGear = RoundMotox( dDBInputSpeed, MMSystem, 'DREHZAHL' );
    oMotox.dOutputSpeed = oMotox.GearData.ABTRIEBSDREHZAHL - 0;
    oMotox.dServiceFactor = oMotox.GearData.BETRIEBSFAKTOR - 0;
}

function _adjustTechnicalData_Adapter( oMotox, oParsingData, oShortDescriptions )
{
    // gear data from parsing
    var dDBRatio = oParsingData.oRatio.EGUEB;
    var dDBNominalTorque = oParsingData.oRatio.T2FB1;
    var MMSystem = "SI";
    
    // technical data to calculate
    //hier Berechnung der Antriebs/Abtriebswerte
    
    var nStufenFaktor    = 1.0;
    var nVerlustLeistung = 0;
    var etaPab = 1.0;
    
    var dDBInputSpeed, dDBOutputSpeed, dDBPower, dDBOutputTorque;
    
    dDBInputSpeed = oMotox.dInputSpeed;
    dDBOutputSpeed = dDBInputSpeed/dDBRatio;
    
    var dDBPower = oMotox.strMotorPowerGA;
    
    //Bei Schneckenstirnradgetrieben wird Wirkungsgrad gesondert berechnet
    if(oMotox.GearData.EGBEZTYP == "C")
    {
        etaPab = getWirkungsgradSchneckengetriebe( oMotox.GearData.EGBEZTYP, oMotox.GearData.EGBEZBRA, oMotox.GearData.EGBEZUEBS, oMotox.GearData.EGBEZEI, oMotox.GearData.EGBEZDIM, "-", "-", "-", dDBRatio, dDBOutputSpeed, MMSystem);
    }
    
    var pAB = (dDBPower - nVerlustLeistung) * nStufenFaktor * etaPab;
    
    dDBOutputTorque = ( 9550.0 * pAB ) / dDBOutputSpeed;
    
    //Bei Schneckenstirnradgetrieben wird das Nennmoment gesondert berechnet
    if(oMotox.GearData.EGBEZTYP == "C")
    {
        dDBNominalTorque = getNennmomentSchneckengetriebe( -1, oMotox.GearData.EGBEZTYP, oMotox.GearData.EGBEZBRA, oMotox.GearData.EGBEZUEBS, oMotox.GearData.EGBEZEI, oMotox.GearData.EGBEZDIM, "-", "-", "-", dDBRatio, dDBOutputSpeed, MMSystem);
    }
    
    // actual geared motor technical data
    //var strMMSystem = Settings.getMeasurementSystem();

    //Leistung kommt immer in KW
    
    // Wenn Context vorhanden ist, dann nehmen wir den
    var strMotorRatedPower = ECL_GetValue("ECSI_CX_MotorRatedPower");
    var strMotorRatedSpeed = ECL_GetValue("ECSI_CX_MotorRatedSpeed");
    
    //for Testing only
    if(!strMotorRatedPower)
    {
        strMotorRatedPower = "1,5";
        ECL_SetValue("ECSI_CX_MotorRatedPower",strMotorRatedPower);
    }
    
    if(!strMotorRatedSpeed)
    {
        strMotorRatedSpeed = "1250";
        ECL_SetValue("ECSI_CX_MotorRatedSpeed",strMotorRatedSpeed);
    }

    if(strMotorRatedPower && strMotorRatedSpeed)
    {
        oMotox.GearData.LEISTUNG = strMotorRatedPower;
        oMotox.GearData.ANTRIEBSDREHZAHL = strMotorRatedSpeed;
    }
    else
    {
        oMotox.GearData.LEISTUNG = dDBPower;
        oMotox.GearData.ANTRIEBSDREHZAHL = dDBInputSpeed;
    }
    
    var oLeistungAusgabe = createEmptyTechval();
    oLeistungAusgabe = checkObject( null, 'LEISTUNG', replace( dDBPower, ',', '.' ), 'kW', false );
    
    var strLeistung_Ausgabe;
    if(MMSystem=="SI")
        strLeistung_Ausgabe = getValueFormatted (oLeistungAusgabe, "kW", false, false);
    else
        strLeistung_Ausgabe = getValueFormatted (oLeistungAusgabe, "hp", false, false);
    
    oMotox.GearData.LEISTUNG_AUSGABE = strLeistung_Ausgabe;
    
    
    var strNennmoment_Ausgabe;
    strNennmoment_Ausgabe = RoundMotox( dDBNominalTorque,MMSystem,"DREHMOMENT" );
    
    oMotox.GearData.BDRHZAHL_MOTOR = RoundMotox( dDBInputSpeed, MMSystem, 'DREHZAHL' );
    oMotox.GearData.ABTRIEBSDREHZAHL = RoundMotox( dDBOutputSpeed, MMSystem, 'DREHZAHL' );
    oMotox.GearData.ABTRIEBSDREHZAHL_AUSGABE = replace( oMotox.GearData.ABTRIEBSDREHZAHL, '.', ',' );
    oMotox.GearData.NENNMOMENT = parseFloat(dDBNominalTorque);
    oMotox.GearData.NENNMOMENT_2 = parseFloat(dDBNominalTorque);
    oMotox.GearData.NENNMOMENT_AUSGABE = strNennmoment_Ausgabe;
    
    oMotox.GearData.UEBERSETZUNG = dDBRatio;
    oMotox.GearData.ABTRIEBSDREHMOMENT = RoundMotox( dDBOutputTorque, MMSystem, 'DREHMOMENT' );
    
    
        
    var Abtriebsdrehmoment_Ausgabe;
    Abtriebsdrehmoment_Ausgabe = RoundMotox( dDBOutputTorque,MMSystem,"DREHMOMENT" );
    
    if(MMSystem=="SI"){
        oMotox.GearData.ABTRIEBSDREHMOMENT_AUSGABE = replace( Abtriebsdrehmoment_Ausgabe, '.', ',' );
    }
    else{
        oMotox.GearData.ABTRIEBSDREHMOMENT_AUSGABE = replace( Abtriebsdrehmoment_Ausgabe, ',', '.' );
    }
    oMotox.GearData.BETRIEBSFAKTOR = RoundMotox( dDBNominalTorque / dDBOutputTorque, MMSystem, 'BETRIEBSFAKTOR' );
    
    if(MMSystem=="SI"){
        oMotox.GearData.BETRIEBSFAKTOR_AUSGABE = replace( oMotox.GearData.BETRIEBSFAKTOR, '.', ',' );
    }
    else{
        oMotox.GearData.BETRIEBSFAKTOR_AUSGABE = replace( oMotox.GearData.BETRIEBSFAKTOR, ',', '.' );
    }

    // we set the input values for the geared motor on the actual technical values; with this settings, the geared motor will be found as a possible result in a subseqent search 
    // gear input torque
    oMotox.bMotorPowerOrOutputTorque = true;

    // dDBTorque war undefined, um dies zu umgehen wurde es auf 0 gesetzt. Kein rückschluss woher die Variable kommen
    // könnte, da in der alten engine auch undefined ist
    var dDBTorque = 0;

    if ( !oMotox.oOutputTorque )
    {
        oMotox.oOutputTorque = createEmptyTechval();
        oMotox.oOutputTorque = checkObject( null, 'DREHMOMENT', dDBTorque, 'Nm',false );
    }
    else setValueAsNumber( oMotox.oOutputTorque, dDBTorque, 'Nm', true );
    if ( !oMotox.oGearNominalTorque )
    {
        oMotox.oGearNominalTorque = createEmptyTechval();
        oMotox.oGearNominalTorque = checkObject( null, 'DREHMOMENT', dDBNominalTorque, 'Nm',false );
    }
    else setValueAsNumber( oMotox.oGearNominalTorque, dDBNominalTorque, 'Nm', true );

    // motor input power
    oMotox.strMotorPower = '' + dDBPower;
    if ( !oMotox.oMotorPower )
    {
        oMotox.oMotorPower = createEmptyTechval();
        oMotox.oMotorPower = checkObject( null, 'LEISTUNG', replace( oMotox.strMotorPower, ',', '.' ), 'kW',false );
    }
    else setValueAsNumber( oMotox.oMotorPower, replace( oMotox.strMotorPower, ',', '.' ), 'kW', true );
    
    oMotox.dRatio = dDBRatio;
    oMotox.strInputSpeedGear = RoundMotox( dDBInputSpeed, MMSystem, 'DREHZAHL' );
    oMotox.dOutputSpeed = oMotox.GearData.ABTRIEBSDREHZAHL - 0;
    oMotox.dServiceFactor = oMotox.GearData.BETRIEBSFAKTOR - 0;
}


/**
 * @constructor
 * build a geared motor object from parsing data
 *
 * @param oParsingData - results from the parsing process
 *                       oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *								    EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                       oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                       oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                       oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                       oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *         					     MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                               WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, SUCHTYP etc. (all columns of table CD_MOTOREN)
 * @param oShortDescriptions - object for short descriptions (and optional corresponding freetext)
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group and freetext (the freetext string may be empty)
 * @param strBaseMLFB - base MLFB
 *
 * @return object - constructed geared motor object
 */
function buildGearedMotor( oParsingData, oShortDescriptions, strBaseMLFB, strRegion )
{
    var oMotox = {};
    
    oMotox.GearData = {};
    oMotox.oConfig = {};
    oMotox.BrakeData = {};
    //--------------------------------------------------------------------------------------------------------
    // base geared motor data
    //--------------------------------------------------------------------------------------------------------
    
    // base gear and motor stuff
    oMotox.EGTYP = oParsingData.oGearType.EGTYP;
    
    /*
    if(oParsingData.oGearType.ANWTYP=="KleinSchnecke")
        oMotox.strMotor = replace((oParsingData.oMotor.MOTTYP+oParsingData.oMotor.MOEINKL+"I"+oParsingData.oMotor.MOTDIM+oParsingData.oMotor.PAKZUS+oParsingData.oMotor.PAKLGE+oParsingData.oMotor.POLZAHL+oParsingData.oMotor.HEMOTOR+oParsingData.oMotor.VSDMOTOR),"-","");
    else
        oMotox.strMotor = replace((oParsingData.oMotor.MOTTYP+oParsingData.oMotor.MOEINKL+oParsingData.oMotor.MOTDIM+oParsingData.oMotor.PAKZUS+oParsingData.oMotor.PAKLGE+oParsingData.oMotor.POLZAHL+oParsingData.oMotor.HEMOTOR+oParsingData.oMotor.VSDMOTOR),"-","");
    */
    
    oMotox.strMotor = oParsingData.oMotor.MOT_BZ_LANG;
    oMotox.strProduct = oParsingData.strProduct;
    oMotox.ECSI_2KJ_TYPE = oParsingData.strProduct;
    oMotox.ECSI_G115_TYPE = oParsingData.strG115;
    
    // the frequency in the parsing results sets the net frequency of the motor; the gear attribut is the motor frequency
    oMotox.ECSI_FREQUENZ_VORAUSWAHL	= oParsingData.oMotor.strFrequency;
    
    if ( oParsingData.oGearType.EGBEZTYP === '-' )
        oMotox.ECSI_GETRIEBETYP = 'Stirnradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'F' )
        oMotox.ECSI_GETRIEBETYP = 'Flachgetriebe';
    else if ( ( oParsingData.oGearType.EGBEZTYP === 'K' ) || ( oParsingData.oGearType.EGBEZTYP === 'B' ) )
        oMotox.ECSI_GETRIEBETYP = 'Kegelradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'C' )
        oMotox.ECSI_GETRIEBETYP = 'Schneckenstirnradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'S' )
        oMotox.ECSI_GETRIEBETYP = 'Schneckengetriebe';
    
    oMotox.strMotorType = '1' + oMotox.strMotor.substring( 0, 2 );
    oMotox.strMotorSize	= oParsingData.oMotor.MOTDIM;
    oMotox.strPoles = oParsingData.oMotor.POLZAHL; 		// take care: motor poles may be overridden by a short description
    oMotox.bWelle2Polig = ( oMotox.strPoles === '2' );
    // leave the default for efficiency class
    
    oMotox.GearData.MOT_EFFIZIENZKL_KNZ = oParsingData.oMotor.strEFFKLASSE;	
    
    oMotox.strSelectedGearID = beautifyMLFB16( strBaseMLFB );
    
    //Antriebsflanschdurchmnesser bestimmen
    var strMOABFLDM;
    
    var strGetriebe_Bez = oParsingData.oGearType.EGBEZTYP+oParsingData.oGearType.EGBEZUEBS+oParsingData.oGearType.EGBEZDIM;
    strGetriebe_Bez = replace(strGetriebe_Bez, "-", "");
    
    var strSelect = "SELECT EGANFLDM FROM CD_GETRIEBE_BEZEICHNUNGEN WHERE ";
    strSelect = strSelect + "GETRIEBE_BEZ = '"+strGetriebe_Bez+"' ";
    
    strSelect = strSelect + "AND CD_GETRIEBE_BEZEICHNUNGEN.REGION = 'EUR' ";
    
    var strDBResult = 'PARSER_GETRIEBEDATA';
    if ( dbselect( '2KJ8_DB', strSelect, strDBResult ) > 0 )
    {
        strMOABFLDM = dbgetstring( strDBResult, 'EGANFLDM' );
    }
    else
    {
        strMOABFLDM = "";
    }
    
    // pegasus attributes
    oMotox.GearData.EGBEZTYP = oParsingData.oGearType.EGBEZTYP;
    oMotox.GearData.EGBEZUEBS = oParsingData.oGearType.EGBEZUEBS;
    oMotox.GearData.EGBEZBRA = oParsingData.oGearType.EGBEZBRA;
    oMotox.GearData.ANWTYP = oParsingData.oGearType.ANWTYP;
    oMotox.GearData.EGBEZDIM = oParsingData.oGearType.EGBEZDIM;
    oMotox.GearData.EGBEZEI = oParsingData.oGearType.EGBEZEI;
    oMotox.GearData.ZGBEZUEBS = oParsingData.oGearType.ZGBEZUEBS;
    oMotox.GearData.ZGBEZDIM = oParsingData.oGearType.ZGBEZDIM;
    oMotox.GearData.ZGBEZEI = oParsingData.oGearType.ZGBEZEI;
    oMotox.GearData.PAKLGE = oParsingData.oMotor.PAKLGE;
    oMotox.GearData.BEZEICHNUNG = oMotox.EGTYP + '-' + oMotox.strMotor;
    oMotox.GearData.MOTDIM = oParsingData.oMotor.MOTDIM;
    oMotox.GearData.MOTBEZ = oMotox.strMotor;
    oMotox.GearData.MOT_BZ_LANG = oMotox.strMotor;
    oMotox.GearData.MOTTYP = oMotox.strMotor.substring( 0, 2 );
    oMotox.GearData.MOSHLT = oParsingData.oMotor.SCHALTUNG_S;
    oMotox.GearData.MOABFLDM = strMOABFLDM;
    oMotox.GearData.EGANFLDM = strMOABFLDM;
    oMotox.GearData.PAKLGE = oParsingData.oMotor.PAKLGE;
    oMotox.GearData.FREQUENZ = oParsingData.oMotor.strFrequency;
    oMotox.oConfig.WAERMEKLASSE = oParsingData.oMotor.MOTWK;
    oMotox.GearData.WIBLNR = oParsingData.oMotor.WIBLNR;
    oMotox.GearData.WIBLAUSLG = oParsingData.oMotor.WIBLAUSLG;
    
    //Zusatzdaten, um die MLFB des Motors zu bestimmen
    oMotox.GearData.MOTFHT_FREQUENZ = oParsingData.oMotor.FREQUENZ;
    oMotox.GearData.MOTFHT_FREQUENZ2 = oParsingData.oMotor.FREQUENZ2;
    oMotox.GearData.MOTFHT_LEISTUNG1 = oParsingData.oMotor.LEISTUNG1;
    oMotox.GearData.MOTFHT_LEISTUNG4 = oParsingData.oMotor.LEISTUNG4;
    oMotox.GearData.MOTFHT_LEISTUNG21 = oParsingData.oMotor.LEISTUNG21;
    oMotox.GearData.MOTFHT_LEISTUNG24 = oParsingData.oMotor.LEISTUNG24;
    oMotox.GearData.MOTFHT_EFFKLASSE = oParsingData.oMotor.EFFKLASSE;
    oMotox.GearData.MOTFHT_HEMOTOR = oParsingData.oMotor.HEMOTOR;
    oMotox.GearData.MOTELAUSF = oParsingData.oMotor.MOTELAUSF;
    oMotox.GearData.SPEC_STRING= createSpecStringParser(oMotox.ECSI_G115_TYPE);
    
    if(oParsingData.oGearType.ANWTYP=="KleinSchnecke")
        oMotox.GearData.S09="1";
    else
        oMotox.GearData.S09="0";
    
    oMotox.GearData.VSDMOTOR=oParsingData.oMotor.VSDMOTOR;
    
    if(oMotox.GearData.VSDMOTOR=="V")
        oMotox.oConfig.ECSI_VSD_MOTOREN="VSD_4000";
        
    if(oMotox.GearData.EGBEZBRA == "H")
        oMotox.oConfig.ECSI_BRANCHENANTRIEBE="HAENGEBAHN";
    
    oMotox.GearData.M3 = oParsingData.oMotor.M3;
    if(parseInt(oMotox.GearData.MOTDIM) >= 225)
        oMotox.GearData.GM3 = "1";
    else
        oMotox.GearData.GM3 = "0";
    
    oMotox.GearData.BRE_DEFAULT = oParsingData.oMotor.BRE_DEFAULT;
    
    
    // technical data
    _adjustTechnicalData( oMotox, oParsingData, strRegion );
    
    //--------------------------------------------------------------------------------------------------------
    // configuration
    //--------------------------------------------------------------------------------------------------------	
    
    oMotox.oConfig.BETRIEBSART = oParsingData.oMotor.MOTBART;
        
    //irrelevant für 2KJ8
    oMotox.oConfig.ECSI_MOTORSPANNUNG_SPGTOL_KA = "-";
    oMotox.oConfig.ECSI_MOTORSPANNUNG_BART_KA = "-";
    oMotox.oConfig.ECSI_MOTORSPANNUNG_P_KA = "-";
    oMotox.oConfig.ECSI_MOTORSPANNUNG_SPG_KA = "-";
    
    //13. Stelle der MLFB
    oMotox.oConfig.ECSI_MOTORSPANNUNG_SPG = oParsingData.oMotor.strSpannung;
    oMotox.oConfig.ECSI_PAKLGE = oParsingData.oMotor.strPAKLGE;
    oMotox.oConfig.ECSI_MOTORSPANNUNG_SPG_MLFB = oParsingData.oMotor.MLFB_2KJ8_13;
    
    //Spannung zusammenbauen
    
    var outputSpannung="";
    
    var strSpannung1 = oParsingData.oMotor.MOTSPG_1;
    var strSpannung2 = oParsingData.oMotor.MOTSPG_3;
    var strSpannung3 = oParsingData.oMotor.MOTSPG_5;
    var strSpannung4 = oParsingData.oMotor.MOTSPG_7;
    
    var strSpannung21 = oParsingData.oMotor.MOTSPG2_1;
    var strSpannung22 = oParsingData.oMotor.MOTSPG2_3;
    var strSpannung23 = oParsingData.oMotor.MOTSPG2_5;
    var strSpannung24 = oParsingData.oMotor.MOTSPG2_7;
    
    var strSchaltung11 = oParsingData.oMotor.MOTSHLT1_1;
    var strSchaltung12 = oParsingData.oMotor.MOTSHLT12_1;
    var strSchaltung21 = oParsingData.oMotor.MOTSHLT2_1;
    var strSchaltung22 = oParsingData.oMotor.MOTSHLT22_1;
    
    var teilschaltung11a="";
    var teilschaltung11b="";
    
    var teilschaltung12a="";
    var teilschaltung12b="";
    
    var teilschaltung21a="";
    var teilschaltung21b="";
    
    var teilschaltung22a="";
    var teilschaltung22b="";
    
    var i = find(strSchaltung11, "/");
    if(i > -1)
    {
        teilschaltung11a = strSchaltung11.left(i-1);
        teilschaltung11b = strSchaltung11.right(strSchaltung11.length-i);
    }
    
    var i = find(strSchaltung12, "/");
    if(i > -1)
    {
        teilschaltung12a = strSchaltung12.left(i-1);
        teilschaltung12b = strSchaltung12.right(strSchaltung12.length-i);
    }
    
    var i = find(strSchaltung21, "/");
    if(i > -1)
    {
        teilschaltung21a = strSchaltung21.left(i-1);
        teilschaltung21b = strSchaltung21.right(strSchaltung21.length-i);
    }
    
    var i = find(strSchaltung22, "/");
    if(i > -1)
    {
        teilschaltung22a = strSchaltung22.left(i-1);
        teilschaltung22b = strSchaltung22.right(strSchaltung22.length-i);
    }
    
    if(oParsingData.oMotor.strFrequency == "50")
    {
        if(oParsingData.oMotor.FREQUENZ_1 == "50")
        {
            if(strSpannung2 != "-")
            {
                outputSpannung = strSpannung1+ " V"+teilschaltung11a+ " / " + strSpannung2 + " V"+teilschaltung11b;
            }
            else	
                outputSpannung = strSpannung1+ " V"+strSchaltung11;
        }
    }
    else if(oParsingData.oMotor.strFrequency == "60")
    {
        if(oParsingData.oMotor.FREQUENZ_1 == "60")
        {
            if(strSpannung2 != "-")
            {
                outputSpannung = strSpannung1+ " V"+teilschaltung11a+ " / " + strSpannung2 + " V"+teilschaltung11b;
            }
            else	
                outputSpannung = strSpannung1+ " V"+strSchaltung11;
        }
        else if(oParsingData.oMotor.FREQUENZ_2 == "60")
        {
            if(strSpannung4 != "-")
            {
                outputSpannung = strSpannung3+ " V"+teilschaltung12a+ " / " + strSpannung4 + " V"+teilschaltung12b;
            }
            else	
                outputSpannung = strSpannung3+ " V"+strSchaltung12;
        }
    }
    else if(oParsingData.oMotor.strFrequency == "87")
    {
        if(oParsingData.oMotor.FREQUENZ_2 == "87")
        {
            if(strSpannung4 != "-")
            {
                outputSpannung = strSpannung3+ " V"+teilschaltung12a+ " / " + strSpannung4 + " V"+teilschaltung12b;
            }
            else	
                outputSpannung = strSpannung3+ " V"+strSchaltung12;
        }
        else if(oParsingData.oMotor.FREQUENZ2_1 == "87")
        {
            if(strSpannung22 != "-")
            {
                outputSpannung = strSpannung21+ " V"+teilschaltung21a+ " / " + strSpannung22 + " V"+teilschaltung21b;
            }
            else	
                outputSpannung = strSpannung21+ " V"+strSchaltung21;
        }
    }
    else if(oParsingData.oMotor.strFrequency == "104")
    {
        if(oParsingData.oMotor.FREQUENZ2_2 == "104")
        {
            if(strSpannung24 != "-")
            {
                outputSpannung = strSpannung23+ " V"+teilschaltung22a+ " / " + strSpannung24 + " V"+teilschaltung22b;
            }
            else	
                outputSpannung = strSpannung23+ " V"+strSchaltung22;
        }
    }
    

    oMotox.oConfig.MOTORSPANNUNG = outputSpannung;
    oMotox.oConfig.WIBLAUSLG = oParsingData.oMotor.WIBLAUSLG;
    
    //Feststellen, ob erhöhte Leistung vorliegt, über DTK Tabellen
    /*
    var strERHL_Select="";
    strERHL_Select = "SELECT DISTINCT MOT_LEISTUNG_ERH_TF FROM TMP_MOT_DATEN WHERE ";
    strERHL_Select = strERHL_Select + "MOT_FREQUENZ='"+oParsingData.oMotor.strFrequency+"' AND ";
    strERHL_Select = strERHL_Select + "MOT_SPANNUNG_UI='"+outputSpannung+"' AND ";
    strERHL_Select = strERHL_Select + "MOT_BETRIEBSART='"+oParsingData.oMotor.MLFB_2KJ_BART_KA+"' AND ";
    
    strERHL_Select = strERHL_Select + "MLFB_10='"+oParsingData.oMotor.MLFB_2KJ_10+"' AND ";
    strERHL_Select = strERHL_Select + "MLFB_13='"+oParsingData.oMotor.MLFB_2KJ_13+"' AND ";
    strERHL_Select = strERHL_Select + "MLFB_13_KA='"+oParsingData.oMotor.MLFB_2KJ_13_KA+"' AND ";
    strERHL_Select = strERHL_Select + "MOT_EL_AUSFUEHRUNG='"+oParsingData.oMotor.MOTELAUSF+"' AND ";
    strERHL_Select = strERHL_Select + "MOT_EFFIZIENZKL_KNZ='"+oMotox.GearData.MOT_EFFIZIENZKL_KNZ+"' AND ";
    strERHL_Select = strERHL_Select + "MOT_LEISTUNG='"+oMotox.GearData.LEISTUNG+"' ";
    var aResultERHL_Leistung = Object.fromDatabase( '2KJ8_DB', strERHL_Select );	
    if(aResultERHL_Leistung.length == 1)
    {
        if(aResultERHL_Leistung[0].MOT_LEISTUNG_ERH_TF == "Nein")
            oMotox.oConfig.ERHOEHTE_LEISTUNG_OPTION = "F";
        else if(aResultERHL_Leistung[0].MOT_LEISTUNG_ERH_TF == "Ja")
            oMotox.oConfig.ERHOEHTE_LEISTUNG_OPTION = "T";
    }
    else
        logtraceln("Fehler Bestimmung erhöhte Leistung!");
    */
    
    //G115D Attribute setzen
    if(oParsingData.oG115D)
    {
        oMotox.oConfig.MOTORSTECKER=oParsingData.oG115D.KKMAT;
        oMotox.oConfig.GEGENSTECKER=oParsingData.oG115D.KKMAT_ZUB ;
        oMotox.oConfig.G115_KOMMUNIKATION=oParsingData.oG115D.G115_BUSSYSTEM;
        oMotox.oConfig.G115_ANSCHLUSS=oParsingData.oG115D.G115_BUSSYSTEM_VS;
        oMotox.oConfig.G115_POSITION_KOMM=oParsingData.oG115D.G115_BUSSYSTEM_POS;
        oMotox.oConfig.G115_ANSCHLUSSTECHNIK=oParsingData.oG115D.G115_ANSCHLUSSTECHNIK_2KJ8;
    }	
    
    _adjustGearedMotorWithShortDescriptions( oShortDescriptions, oMotox, oParsingData, strRegion  );
    
    if(getSDFromGroup( oShortDescriptions, 'GETRIEBE_BEFESTIGUNGSART' ) == ""){
        oMotox.oConfig.BEFESTIGUNGSART = "-";
    }
    
    var oShaft=GetAbtriebswelle_2KJ8( oMotox, oParsingData.oGearType, oShortDescriptions);
    
    if(oShaft)
    {
        oMotox.GearData.EGBEZAWTYP = oShaft.EGBEZAWTYP;
        oMotox.GearData.EGBEZWNV = oShaft.EGBEZWNV;
        oMotox.GearData.EGTYP = oShaft.EGTYP;
        
        // Abtriebswellenmass
        oMotox.oConfig.ABTRIEBSWELLENMASS = replace(oShaft.EGABW,'"','ZOLL');
        
        var strWelle = oShaft.EGABKDBEZ;
        if(strWelle)
            strWelle = strWelle.replace(/[^a-zA-Z]/g, '');
        
        var strWelle_EGABAUSF = oShaft.EGABAUSF;
        var strWelle_WELLE_BDS = oShaft.WELLE_BDS;
        var strWelle_WELLE_O_PF = oShaft.WELLE_O_PF;
        
        if(strWelle=="H" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle";
        
        if(strWelle=="HF" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_SIMOLOC";
        
        if(strWelle=="HM" && strWelle_EGABAUSF=="VLplus" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle";
        
        if(strWelle=="HS" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_Schrumpfscheibe";
            
        if(strWelle=="HX" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_Edelstahl";
            
        if(strWelle=="N" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_Vielkeilverzahnung";
            
        if(strWelle=="V" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_mit_Passfeder";
            
        if(strWelle=="VC" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_fuer_Kuehlturmantriebe";
            
        if(strWelle=="VD" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="1" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_beidseitig";
            
        if(strWelle=="VDG" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="1" && strWelle_WELLE_O_PF=="1")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_beidseitig_ohne_Passfeder";
            
        if(strWelle=="VE" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Einsteckwelle";
        
        if(strWelle=="VG" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="1")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_ohne_Passfeder";
            
        if(strWelle=="VM" && (strWelle_EGABAUSF=="VLplus" || strWelle_EGABAUSF=="XLplus") && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_mit_Passfeder";
        
        if(strWelle=="W" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
            oMotox.oConfig.ABTRIEBSWELLENART = "Welle_fuer_SIP";
        
        
        var strFlangeSD = getSDFromGroup( oShortDescriptions, 'FLANSCHDURCHMESSER' );
        // for an empty short description, we get the default for this gear, if the fastening fits
        var strFlangeDiameter = getFlangeDiameter( strFlangeSD, oParsingData.oGearType.EGBEZTYP, oParsingData.oGearType.EGBEZUEBS, oParsingData.oGearType.EGBEZBRA,oParsingData.oGearType.EGBEZDIM, oParsingData.oGearType.EGBEZEI, oMotox.oConfig.BEFESTIGUNGSART );
        if ( strFlangeDiameter !== '' )
            oMotox.oConfig.BEFESTIGUNGSMASS = strFlangeDiameter;
            
        
        var strFlangeFigur=getFlangeFigur( strFlangeSD, oParsingData.oGearType.EGBEZTYP, oParsingData.oGearType.EGBEZUEBS, oParsingData.oGearType.EGBEZBRA,oParsingData.oGearType.EGBEZDIM, oParsingData.oGearType.EGBEZEI, oMotox.oConfig.BEFESTIGUNGSART );
        if ( strFlangeFigur !== '' )
            oMotox.oConfig.FIGUR = strFlangeFigur;
    }	
    
    oMotox.oConfig.MOTORSCHUTZ = oParsingData.MotorProtection;
    
    //Klemmkastenlage bei MotorMounted anpassen
    if(oMotox.ECSI_G115_TYPE == "MM")
    {
        var strKlemmkastenlage=oMotox.oConfig.KLEMMKASTENLAGE;
        oMotox.oConfig.KLEMMKASTENLAGE=left(oMotox.oConfig.KLEMMKASTENLAGE,1);
    }
    
    if(oParsingData.Brake == "mit")
        oMotox.oConfig.MIT_BREMSE = "T";
    else
        oMotox.oConfig.MIT_BREMSE = "";
    
    // synchronize gear brake data with brake data from configuration object
    if ( oMotox.oConfig.MIT_BREMSE == "T" )
    {
        var strBrake="";
        oMotox.BrakeData.BREART = oParsingData.oBrake.BREART;
        oMotox.BrakeData.BRENENMO = oParsingData.oBrake.BRENENMO;
        oMotox.BrakeData.BREREDMO = oParsingData.oBrake.BREREDMO;

        if(oMotox.BrakeData.BREART == "FDX")
        {
            strBrake = aResult[0].BREGRDTYP+"/"+aResult[0].BREREDMO;
        }
        else if(oMotox.BrakeData.BREREDMO == "-")
        {
            strBrake = oMotox.BrakeData.BREART+oMotox.BrakeData.BRENENMO;
        }
        else
        {
            strBrake = oMotox.BrakeData.BREART+oMotox.BrakeData.BRENENMO+"/"+oMotox.BrakeData.BREREDMO;
        }
       
        oMotox.oConfig.BREMSENTYP = strBrake;
        
        if ( 'C20' in oShortDescriptions)
            oMotox.BrakeData.BRESHLT = 'am Klemmbrett';
        else // if ( 'C21' in oShortDescriptions ) // is also the default
            oMotox.BrakeData.BRESHLT = 'nicht am Klemmbrett';
    }
    
    oMotox.ECSI_POLZAHL = oParsingData.oMotor.POLZAHL; 		// motor poles may be overridden by a short description
    
    // Sonderlogik 
	if(oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_OPTION == "T")
	{
		if(oMotox.GearData.EGBEZTYP=="B" || oMotox.GearData.EGBEZTYP=="K")
		{
			if(oMotox.oConfig.BAUFORM=="M4")
			{
				if(!oMotox.oConfig.OELSTANDSKONTROLLE || oMotox.oConfig.OELSTANDSKONTROLLE=="ohne" || oMotox.oConfig.OELSTANDSKONTROLLE=="-")
				{
					oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20";
				}
				else
				{
					oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20-0";
				}
			}
			else
			{
					oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20";
			}
		}
		else
		{
			oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "X_20";
		}
	}
	
    return( oMotox );
}

function buildGear_ADAPTER( oParsingData, oShortDescriptions, strBaseMLFB, strRegion )
{
    var MMSystem="SI";
    
    var oMotox = {};
    
    oMotox.GearData = {};
    oMotox.oConfig = {};
    oMotox.BrakeData = {};
    
    //--------------------------------------------------------------------------------------------------------
    // base geared motor data
    //--------------------------------------------------------------------------------------------------------
    
    // base gear and motor stuff
    oMotox.strGear = oParsingData.oShaft.EGTYP;
    oMotox.EGTYP = oParsingData.oGearType.EGTYP;
    oMotox.strProduct = oParsingData.strProduct;
    oMotox.ECSI_2KJ_TYPE = oParsingData.strProduct;
    
    // the frequency in the parsing results sets the net frequenc<y of the motor; the gear attribut is the motor frequency
    oMotox.strFrequency	= '';	
    oMotox.strAdapter = oParsingData.oMotor.AV_TYP_KZ;
    oMotox.strDriveGroup = oParsingData.oMotor.AV_BEZ;
    oMotox.strShaftHeight = oParsingData.oMotor.AV_GROESSE_KZ;
    
    if ( oParsingData.oGearType.EGBEZTYP === '-' )
        oMotox.ECSI_GETRIEBETYP = 'Stirnradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'F' )
        oMotox.ECSI_GETRIEBETYP = 'Flachgetriebe';
    else if ( ( oParsingData.oGearType.EGBEZTYP === 'K' ) || ( oParsingData.oGearType.EGBEZTYP === 'B' ) )
        oMotox.ECSI_GETRIEBETYP = 'Kegelradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'C' )
        oMotox.ECSI_GETRIEBETYP = 'Schneckenstirnradgetriebe';
    else if ( oParsingData.oGearType.EGBEZTYP === 'S' )
        oMotox.ECSI_GETRIEBETYP = 'Schneckengetriebe';

    if ( 'N30' in oShortDescriptions )
    {
        // specification EAC
        oMotox.strSpecification = 'EAC';
    }
    else 
        oMotox.strSpecification = 'CE';
    
    var strGA_Bez;
    if(oParsingData.oMotor.AV_RLSP == '1' && (oParsingData.oMotor.AV_TYP_KZ == "K2" || oParsingData.oMotor.AV_TYP_KZ == "K3"))
    {
        strGA_Bez = oParsingData.oMotor.AV_TYP_KZ+"X-"+oParsingData.oMotor.AVTYPDIM;
    }
    else
        strGA_Bez = oParsingData.oMotor.AV_TYP_KZ+"-"+oParsingData.oMotor.AVTYPDIM;
    
    oMotox.strMotorType = '';
    oMotox.strMotor = strGA_Bez;
    oMotox.strMotorSize	= oParsingData.oMotor.AVTYPDIM;
    oMotox.strPoles = '';
    oMotox.bWelle2Polig = '';
    oParsingData.oMotor.SUCHTYP = strGA_Bez;
    
    oMotox.strEfficiencyClass = '';
    
    oMotox.strSelectedGearID = beautifyMLFB16( strBaseMLFB );
    
    
    var strGetriebe_Bez = oParsingData.oGearType.EGBEZTYP+oParsingData.oGearType.EGBEZUEBS+oParsingData.oGearType.EGBEZDIM;
    strGetriebe_Bez = replace(strGetriebe_Bez, "-", "");
    
    var strAVABFLD;
    var strSelect = "SELECT DISTINCT EGANFLDM FROM EGANBEF WHERE ";
    strSelect = strSelect + "EGANBEF.EGBEZTYP = '"+oParsingData.oGearType.EGBEZTYP+"' AND ";
    strSelect = strSelect + "EGANBEF.EGBEZUEBS = '"+oParsingData.oGearType.EGBEZUEBS+"' AND ";
    strSelect = strSelect + "EGANBEF.EGBEZBRA = '"+oParsingData.oGearType.EGBEZBRA+"' AND ";
    strSelect = strSelect + "EGANBEF.EGBEZDIM = '"+oParsingData.oGearType.EGBEZDIM+"' AND ";
    strSelect = strSelect + "EGANBEF.EGBEZEI = '"+oParsingData.oGearType.EGBEZEI+"' ";
    
    var strDBResult = 'PARSER_GETRIEBEDATA_AVMOKDBEZ';
    var strAVMOKDBEZ;
    
    if ( dbselect( '2KJ8_DB', strSelect, strDBResult ) > 0 )
    {
        strAVABFLD = dbgetstring( strDBResult, 'EGANFLDM' );
    }
    else
    {
        strAVABFLD = "";
    }
    
    var strSelect = "SELECT DISTINCT AVANSSAR, AVABFLD FROM CD_AVEINHEITEN WHERE ";
    strSelect = strSelect + "AV_BEZ = '"+oParsingData.oMotor.AV_BEZ+"' AND ";
    strSelect = strSelect + "AVABFLD = '"+strAVABFLD+"' ";
    
    /*
    if(strRegion == "CHINA")
        strSelect = strSelect + "AND CD_AVEINHEITEN.REGION = 'CHINA' ";
    else if(strRegion == "USA")
        strSelect = strSelect + "AND CD_AVEINHEITEN.REGION = 'USA' ";
    else
        strSelect = strSelect + "AND CD_AVEINHEITEN.REGION = 'EUR' ";
    */
    
    var strAVANSSAR;
    var strDBResult = 'PARSER_GETRIEBEDATA';
    if ( dbselect( '2KJ8_DB', strSelect, strDBResult ) > 0 )
    {
        strAVANSSAR = dbgetstring( strDBResult, 'AVANSSAR' );
    }
    else
    {
        strAVANSSAR = "";
    }
    
    
    // pegasus attributes
    oMotox.GearData.EGBEZTYP = oParsingData.oGearType.EGBEZTYP;
    oMotox.GearData.EGBEZUEBS = oParsingData.oGearType.EGBEZUEBS;
    oMotox.GearData.EGBEZBRA = oParsingData.oGearType.EGBEZBRA;
    oMotox.GearData.ANWTYP = oParsingData.oGearType.ANWTYP;
    oMotox.GearData.EGBEZDIM = oParsingData.oGearType.EGBEZDIM;
    oMotox.GearData.EGBEZEI = oParsingData.oGearType.EGBEZEI;
    oMotox.GearData.ZGBEZUEBS = oParsingData.oGearType.ZGBEZUEBS;
    oMotox.GearData.ZGBEZDIM = oParsingData.oGearType.ZGBEZDIM;
    oMotox.GearData.ZGBEZEI = oParsingData.oGearType.ZGBEZEI;
    oMotox.GearData.EGBEZAWTYP = oParsingData.oShaft.EGBEZAWTYP;
    oMotox.GearData.EGBEZWNV = oParsingData.oShaft.EGBEZWNV;
    oMotox.GearData.EGTYP = oParsingData.oGearType.EGTYP;
    oMotox.GearData.BEZEICHNUNG = oParsingData.oGearType.EGTYP + '-' + strGA_Bez;
    oMotox.GearData.MOTDIM = oParsingData.oMotor.AVTYPDIM;
    oMotox.GearData.MOTBEZ = strGA_Bez;
    oMotox.GearData.MOTTYP = oParsingData.oMotor.AV_TYP_KZ;
    oMotox.GearData.MOSHLT = '';
    oMotox.GearData.MOABSSAR = strAVANSSAR;
    oMotox.GearData.MOABFLDM = strAVABFLD;
    oMotox.GearData.PAKLGE = '';
    oMotox.oConfig.strFigur = oParsingData.oFastening.EGABDMSTFIG;
    

    oMotox.ANTRIEBSGRUPPE_TYP = oParsingData.oMotor.AV_TYP_KZ;
    
    var oTextData = {};
    if("Y00" in oShortDescriptions){
        oTextData = decodeY00( getFreetextValue( oShortDescriptions.Y00.freetext ) );
    }
    //Daten aus Klartextangaben in Object schreiben
    if("strAdapterAntriebsdrehzahl" in oTextData){
        oMotox.dInputSpeed = oTextData.strAdapterAntriebsdrehzahl;
    }
    else{
        oMotox.dInputSpeed = "1400";
    }
    if(oParsingData.oGearType.ANWTYP=="KleinSchnecke"){
        oMotox.GearData.S09="1";
    }
    else{
        oMotox.GearData.S09="0";
    }
    /*
    var oAdapterleistung = createEmptyTechval();
    if(oTextData.strAdapterAntriebsleistung != null)
        oAdapterleistung = checkObject( null, 'LEISTUNG', replace( oTextData.strAdapterAntriebsleistung, ',', '.' ), 'kW', false );
    else
        oAdapterleistung = checkObject( null, 'LEISTUNG', replace( '1.5', ',', '.' ), 'kW', false );
        
    var strAdapterLeistung;
    if(MMSystem=="SI")
        strAdapterLeistung = getValueFormatted (oAdapterleistung, "kW", false, false);
    else	
        strAdapterLeistung = getValueFormatted (oAdapterleistung, "hp", false, false);
    
    oMotox.strMotorPowerGA = strAdapterLeistung;
    */
    
    //Daten aus Klartextangaben in Object schreiben
    if("strAdapterAntriebsleistung" in oTextData){
        oMotox.strMotorPowerGA = oTextData.strAdapterAntriebsleistung;
    }
    else{
        oMotox.strMotorPowerGA = "1.5";
    }
    
    var dRutschmoment
    if("strAdapterRutschkupplung_Drehmomenteingestellt" in oTextData){
        dRutschmoment =  parseFloat(oTextData.strAdapterRutschkupplung_Drehmomenteingestellt);
    }
    
    oMotox.oConfig.oDriveGroupUserSlipTorque =  checkObject(null, 'DREHMOMENT',dRutschmoment, 'Nm', false );
    
    //ATEX Angaben aus Klartext
    if(oTextData.strAdapterMaxAntriebsleistungATEX != null)
        oMotox.oAtex.oMotorPower = checkObject( null, 'LEISTUNG', replace( oTextData.strAdapterMaxAntriebsleistungATEX, ',', '.' ), 'kW', false );
    else
        oMotox.oAtex.oMotorPower = checkObject( null, 'LEISTUNG', '1.5', 'kW', false );
        
    if(oTextData.strAdapterMaxAntriebsDrehzahlATEX != null)
        oMotox.oAtex.oMaxDrehzahl = checkObject( null, 'DREHZAHL', replace( oTextData.strAdapterMaxAntriebsDrehzahlATEX, ',', '.' ), '1/min', false );
    else
        oMotox.oAtex.oMaxDrehzahl = checkObject( null, 'DREHZAHL', '1500', '1/min', false );
        
    if(oTextData.strAdapterMinAntriebsDrehzahlATEX != null)
        oMotox.oAtex.oMinDrehzahl = checkObject( null, 'DREHZAHL', replace( oTextData.strAdapterMinAntriebsDrehzahlATEX, ',', '.' ), '1/min', false );
    else
        oMotox.oAtex.oMinDrehzahl = checkObject( null, 'DREHZAHL', '1500', '1/min', false );
    
    oMotox.oConfig.FIGUR = oParsingData.oFastening.EGABDMSTFIG;
    
    // technical data
    _adjustTechnicalData_Adapter( oMotox, oParsingData, oShortDescriptions );
    
    //--------------------------------------------------------------------------------------------------------
    // configuration
    //--------------------------------------------------------------------------------------------------------	
    oMotox.oConfig.strOperatingMode = '';
    oMotox.oConfig.strVoltage = '';
    
    oMotox.oConfig.ABTRIEBSWELLENMASS	= replace(oParsingData.oShaft.EGABW,'"','ZOLL');
    
    //Abtriebswellenart setzen
    var strWelle = oParsingData.oShaft.EGABKDBEZ;
    if(strWelle)
        strWelle = strWelle.replace(/[^a-zA-Z]/g, '');
    
    var strWelle_EGABAUSF = oParsingData.oShaft.EGABAUSF;
    var strWelle_WELLE_BDS = oParsingData.oShaft.WELLE_BDS;
    var strWelle_WELLE_O_PF =oParsingData.oShaft.WELLE_O_PF;
    
    if(strWelle=="H" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle";
    
    if(strWelle=="HF" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_SIMOLOC";
    
    if(strWelle=="HM" && strWelle_EGABAUSF=="VLplus" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle";
    
    if(strWelle=="HS" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_Schrumpfscheibe";
        
    if(strWelle=="HX" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_Edelstahl";
        
    if(strWelle=="N" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Hohlwelle_mit_Vielkeilverzahnung";
        
    if(strWelle=="V" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_mit_Passfeder";
        
    if(strWelle=="VC" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_fuer_Kuehlturmantriebe";
        
    if(strWelle=="VD" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="1" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_beidseitig";
        
    if(strWelle=="VDG" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="1" && strWelle_WELLE_O_PF=="1")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_beidseitig_ohne_Passfeder";
        
    if(strWelle=="VE" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Einsteckwelle";
    
    if(strWelle=="VG" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="1")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_ohne_Passfeder";
        
    if(strWelle=="VM" && (strWelle_EGABAUSF=="VLplus" || strWelle_EGABAUSF=="XLplus") && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Vollwelle_mit_Passfeder";
    
    if(strWelle=="W" && strWelle_EGABAUSF=="Standard" && strWelle_WELLE_BDS=="0" && strWelle_WELLE_O_PF=="0")
        oMotox.oConfig.ABTRIEBSWELLENART = "Welle_fuer_SIP";
    
    oMotox.oConfig.BEFESTIGUNGSART = oParsingData.oFastening.EGBEZBEF;
    var strFlangeSD = getSDFromGroup( oShortDescriptions, 'FLANSCHDURCHMESSER' );
    // for an empty short description, we get the default for this gear, if the fastening fits
    var strFlangeDiameter = getFlangeDiameter( strFlangeSD, oParsingData.oGearType.EGBEZTYP, oParsingData.oGearType.EGBEZUEBS, oParsingData.oShaft.EGBEZBRA,oParsingData.oGearType.EGBEZDIM, oParsingData.oGearType.EGBEZEI, oParsingData.oFastening.EGBEZBEF );
    if ( strFlangeDiameter !== '' )
        oMotox.oConfig.BEFESTIGUNGSMASS = strFlangeDiameter;
        
    
    var strFlangeFigur=getFlangeFigur( strFlangeSD, oParsingData.oGearType.EGBEZTYP, oParsingData.oGearType.EGBEZUEBS, oParsingData.oShaft.EGBEZBRA,oParsingData.oGearType.EGBEZDIM, oParsingData.oGearType.EGBEZEI, oParsingData.oFastening.EGBEZBEF );
    if ( strFlangeFigur !== '' )
        oMotox.oConfig.FIGUR = strFlangeFigur;
        
    if(oMotox.strAdapter=="K2")
    {
        var strPolzahl=ECL_GetValue("ECSI_CX_MotorPoleCount");
        if(parseInt(oMotox.strShaftHeight )>=225 && strPolzahl=="2")
        {
            oMotox.oConfig.ECSI_2POL_MOTORABTRIEBSWELLE = "T";
        }
    }

    _adjustGearedMotorWithShortDescriptions( oShortDescriptions, oMotox, oParsingData, strRegion  );	

    
    oMotox.strPoles = ''; 		// motor poles may be overridden by a short description
    oMotox.bWelle2Polig = '';

    //ATEX Verarbeitung
    /*
    var oATEX = oMotox.getAtex();
    
    var strATEXSD = Parser.getSDFromGroup( oShortDescriptions, 'ATEX_GETRIEBE' );
    if(strATEXSD == "K80")
    {
        oATEX.strAtexExplosionGroup = "IIB";
    }
    else if(strATEXSD == "K70")
    {
        oATEX.strAtexExplosionGroup = "IIC";
    }
    else if(strATEXSD == "K81")
    {
        oATEX.strAtexExplosionGroup = "IIB";
    }
    else if(strATEXSD == "K82")
    {
        oATEX.strAtexExplosionGroup = "IIC";
    }
    
    var strATEX_ZONE = Parser.getSDFromGroup( oShortDescriptions, 'ATEX_ZONE' );
    if(strATEX_ZONE == "K71")
    {
        oATEX.strAtexZone = "1";
    }
    else if(strATEX_ZONE == "K72")
    {
        oATEX.strAtexZone = "2";
    }
    else if(strATEX_ZONE == "K73")
    {
        oATEX.strAtexZone = "21";
    }
    else if(strATEX_ZONE == "K74")
    {
        oATEX.strAtexZone = "22";
    }
    
    var strATEX_TK = Parser.getSDFromGroup( oShortDescriptions, 'ATEX_TK' );
    if(strATEX_TK == "K75")
    {
        oATEX.bAtexClassT4 = false;
        oATEX.bAtexSurfaceTemp = true;
    }
    else if(strATEX_TK == "K78")
    {
        oATEX.bAtexClassT4 = true;
        oATEX.bAtexSurfaceTemp = false;
    }
    
    var strATEX_UmgebTemp = Parser.getSDFromGroup( oShortDescriptions, 'ATEX_UMGEB_TEMP' );
    if(strATEX_UmgebTemp == "K85")
        oATEX.strAtexTemperature = "+35";
    else if(strATEX_UmgebTemp == "K86")
        oATEX.strAtexTemperature = "+30";
    else if(strATEX_UmgebTemp == "K87")
        oATEX.strAtexTemperature = "+25";
    else if(strATEX_UmgebTemp == "K88")
        oATEX.strAtexTemperature = "+20";
    else
        oATEX.strAtexTemperature = "+40";
    */

    // Sonderlogik 
    if(oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_OPTION == "T")
    {
        if(oMotox.GearData.EGBEZTYP=="B" || oMotox.GearData.EGBEZTYP=="K")
        {
            if(oMotox.oConfig.BAUFORM=="M4")
            {
               	if(!oMotox.oConfig.OELSTANDSKONTROLLE || oMotox.oConfig.OELSTANDSKONTROLLE=="ohne" || oMotox.oConfig.OELSTANDSKONTROLLE=="-")
                {
                    oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20";
                }
                else
                {
                    oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20-0";
                }
            }
            else
            {
                    oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "Y_20";
            }
        }
        else
        {
            oMotox.oConfig.ZULAESSIGE_EINBAULAGENABWEICHUNG_WINKEL= "X_20";
        }
    }

    return( oMotox );
}


function _addAttributesforG110M_TYP( oMotox, strSD, oParsingData, oShortDescriptions )
{
    var strSelect = "SELECT DISTINCT WERT_1,WERT_2,WERT_3 FROM MLFB_OPTIONEN WHERE MLFB_OPTION = '" + strSD + "' ";
    var strDBResult = 'G110M_TYP';
    
    var strG110M_ADAPTGR_KNZ="";
    
    if ( dbselect( 'SIMOGEAR_DB', strSelect, strDBResult ) == 1 )
    {
        oMotox.oConfig.strRatedPower = dbgetstring( strDBResult, 'WERT_1' );
        strG110M_ADAPTGR_KNZ = dbgetstring( strDBResult, 'WERT_2' );
        oMotox.oConfig.strMotorControl = dbgetstring( strDBResult, 'WERT_3' );
        
        oMotox.oConfig.strVoltageNetwork = "400";
    }
}

function _addAttributesforG110M_CU( oMotox, strSD, oParsingData, oShortDescriptions )
{
    var strSelect = "SELECT DISTINCT WERT_1,WERT_2,WERT_3 FROM MLFB_OPTIONEN WHERE MLFB_OPTION = '" + strSD + "' ";
    var strDBResult = 'G110M_CU';
    
    var strG110M_KKMAT_ZUB="";
    
    if ( dbselect( 'SIMOGEAR_DB', strSelect, strDBResult ) == 1 )
    {
        oMotox.oConfig.strConnection = dbgetstring( strDBResult, 'WERT_2' );
        strG110M_KKMAT_ZUB = dbgetstring( strDBResult, 'WERT_1' );
        oMotox.oConfig.strCommunictaion = dbgetstring( strDBResult, 'WERT_3' );
    }
}

function _addAttributesforG110M_INTBRE( oMotox, strSD, oParsingData, oShortDescriptions )
{
    var strSelect = "SELECT DISTINCT WERT_1 FROM MLFB_OPTIONEN WHERE MLFB_OPTION = '" + strSD + "' ";
    var strDBResult = 'G110M_INTBRE';
    
    if ( dbselect( 'SIMOGEAR_DB', strSelect, strDBResult ) == 1 )
    {
        oMotox.oConfig.bIBw = true;
    }
    else
    {
        oMotox.oConfig.bIBw = false;
    }
    
}

function _addAttributesforG110M_EXTANB( oMotox, strSD, oParsingData, oShortDescriptions )
{
    var strSelect = "SELECT DISTINCT WERT_1 FROM MLFB_OPTIONEN WHERE MLFB_OPTION = '" + strSD + "' ";
    var strDBResult = 'G110M_EXTANB';
    
    if ( dbselect( 'SIMOGEAR_DB', strSelect, strDBResult ) == 1 )
    {
        var strG110M_extAnb = dbgetstring( strDBResult, 'WERT_1' );
        oMotox.oConfig.strExternalAdditions = strG110M_extAnb;
    }
    else
    {
        oMotox.oConfig.strExternalAdditions = "ohne"
    }
}

function GetAbtriebswelle_2KJ8( oMotox, oGearType, oShortDescriptions )
{

    var strAbtriebswelleSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_EGABW' );
    
    var strSelect = "SELECT DISTINCT EGTYP.TYPBEZ AS EGTYP, EGABKD_M3.EGABW AS EGABW, EGABKD_M3.EGABKDBEZ AS EGABKDBEZ,EGABKD_M3.WELLE_BDS AS WELLE_BDS,EGABKD_M3.WELLE_O_PF AS WELLE_O_PF,EGABKD_M3.EGABAUSF AS EGABAUSF, EGTYP.EGBEZAWTYP AS EGBEZAWTYP, EGTYP.EGBEZWNV AS EGBEZWNV, " 
    strSelect = strSelect + "EGTYP.EGBEZBRA AS EGBEZBRA, EGTYP.ANWTYP AS ANWTYP, EGTYP.EGBEZSO AS EGBEZSO FROM MLFB8_EGTYP, EGTYP, EGABKD_M3 WHERE ";
    
    strSelect = strSelect + "EGTYP.CDROM_REL = '1' AND EGABKD_M3.G115='1' AND ";
    
    strSelect = strSelect + "MLFB8_EGTYP.MLFB_2KJ8_5 = '" + oGearType.MLFB_2KJ8_5;
    strSelect = strSelect + "' AND MLFB8_EGTYP.MLFB_2KJ8_6 = '" + oGearType.MLFB_2KJ8_6;
    strSelect = strSelect + "' AND MLFB8_EGTYP.MLFB_2KJ8_7 = '" + oGearType.MLFB_2KJ8_7 ;
    strSelect = strSelect + "' AND EGABKD_M3.MLFB8_OPTION = '" + strAbtriebswelleSD;
    strSelect = strSelect + "' AND EGTYP.EGBEZBEF = '" + oMotox.oConfig.BEFESTIGUNGSART; 
    strSelect = strSelect + "' AND EGTYP.EGBEZBRA = '" + oGearType.EGBEZBRA;
    strSelect = strSelect + "' AND MLFB8_EGTYP.EGBEZTYP = EGTYP.EGBEZTYP AND MLFB8_EGTYP.EGBEZUEBS = EGTYP.EGBEZUEBS AND MLFB8_EGTYP.EGBEZDIM = EGTYP.EGBEZDIM AND ";
    strSelect = strSelect + "MLFB8_EGTYP.EGBEZEI = EGTYP.EGBEZEI AND EGTYP.TYPBEZ = EGABKD_M3.EGTYP ";
    
    
    // we also must distinguish if a backstop or backlash design is present for the gear or not
    if ( 'G72' in oShortDescriptions )
        strSelect += " AND EGTYP.EGBEZSO = 'X' ";
    else if ( 'G99' in oShortDescriptions )
        strSelect += " AND EGTYP.EGBEZSO = 'W' ";
    else strSelect += " AND EGTYP.EGBEZSO = '-' ";
    
    var aResult = Object.fromDatabase( '2KJ8_DB', strSelect );
    if ( aResult.length > 0 )
    {
        return (aResult[0]);
    }
    
    return( null );
}