//@legacy
//#mode(full-ecma)

//@import "Parser.ShortDescriptions.js"

//@import "Parser.MOTOX.ShortDescriptions.js"

//@import "Parser.common.js"

//@import "Parser.types.js"

//@import "Parser.tokenizer.js"

//@import "Parser.MOTOX.checks.js"


//@import "Parser.MOTOX.convert.js"



/**
 * @category   SimoGear Parser
 * @package    Parser
 * @copyright  Copyright (c) 2012-2015 Plan Software GmbH (http://www.plansoft.de)
 */


/**
 * @constructor
 * helper parsing routine for SimoGear geared motors that performs the parsing for the base MLFB
 *
 * @param strBaseMLFB - base MLFB to check
 * @param oShortDescriptions - object with short description information
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group and freetext
 *
 * @return object - results from the parsing process
 *                 strError - error message from parsing (empty, if parsing succeeded)
 *                 strWarning - additional warning message (may be nonempty even if parsing succeeded)
 *                 oParsingData - parsing data object: contains parsing data results from different parsing stages (not all subobjects may be present due to parsing errors)
 *                                oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *											 EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                                oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                                oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                                oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                                oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *                  					  MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                                        WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, MOTSPG, SUCHTYP etc. (all columns of table CD_MOTOREN)
 */
 
function parse_init( strMLFB, strFreetext, bnotfromProjectList )
{
    vtTimeStart	= Date.getTime();
    
    var oParsing = {
        strBaseMLFB : '',
        strError : '',
        strWarning : ''
    };
    
    //Sonderlogik, da SIZER keine E-Zeile kennt
    if(strFreetext=="")
    {
        if(find(strMLFB,'{Y80')>-1)
        {
            //Ich gehe davon aus, dass die Y80 ganz am Schluss der MLFB kommt
            strFreetext = strMLFB.right(strMLFB.length-find(strMLFB,'{Y80')+1);
            strMLFB = replace(strMLFB,strFreetext,"")
        }
        
        if(find(strMLFB,'{Y00')>-1)
        {
            //Ich gehe davon aus, dass die Y80 ganz am Schluss der MLFB kommt
            strFreetext = strMLFB.right(strMLFB.length-find(strMLFB,'{Y00')+1)+strFreetext;
            strMLFB = replace(strMLFB,strMLFB.right(strMLFB.length-find(strMLFB,'{Y00')+1),"")
        }
    }
    
    //-------------------------------------------------------------------------------------------------------
    // normalize MLFB so that we have the base mlfb as a 16char string, followed by optional 
    // additional entrys separated by blanks
    //--------------------------------------------------------------------------------------------------------
    strMLFB = trimright( strMLFB, ' ' );
    strMLFB = trimright( strMLFB, '+' );
    strMLFB = trimright( strMLFB, ' ' );
    strMLFB = normalizeMLFB( strMLFB );
    
    var oResult;
    ECL_SetValue("ECSI_MLFB_CAN_CHANGE_2KJ8", "");
    
    var strStartConfiguration=ECL_GetValue("ECSI_IS_START_CONFIGURATION");
    
    if(strStartConfiguration=="" && find(strMLFB.left(16), ".") >= 0)
        strStartConfiguration="1";
    
    if(strStartConfiguration=="1")
    {
        //Entweder kommt eine zur Haupt MLFB unterschiedliche Base Type MLFB aus dem SIZER.
        //Hier sollen dann möglichst viele Attribute beibehalten werden.
        //Die Basetype MLFB kann sich in Getriebe, Motor oder Übersetzung von der Haupt MLFB unterscheiden.
        var strBaseTypeMLFB=ECL_GetValue("ECSI_MLFB_BASE_TYPE");
        if(strBaseTypeMLFB=="" && find(strMLFB.left(16), ".") >= 0)
        {
            strBaseTypeMLFB=strMLFB;
            ECL_SetValue("ECSI_MLFB_BASE_TYPE",strMLFB);
        }
    
        strBaseTypeMLFB = trimright( strBaseTypeMLFB, ' ' );
        strBaseTypeMLFB = trimright( strBaseTypeMLFB, '+' );
        strBaseTypeMLFB = trimright( strBaseTypeMLFB, ' ' );
        strBaseTypeMLFB = normalizeMLFB( strBaseTypeMLFB );
        
        if(find(strMLFB, ".")>-1 && strBaseTypeMLFB!=strMLFB)
        {
            //Fehler 
            //Bei Keep Configuration muss die alte MLFB komplett sein.
            oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_KEEP_CONFIG_FAIL' );
            
            return oParsing;
        }
        
        //test
        //strBaseTypeMLFB="2KJ82052EG..5.G1";
        
        
        var strGearTypeBT_MLFB = "";
        var strMotorTypeBT_MLFB = "";
        var strRatioTypeBT_MLFB = "";
        var strVoltageBT_MLFB = "";
        
        if(strBaseTypeMLFB != "")
        {
            //--------------------------------------------------------------------------------------------------------
            // Base Type  MLFB aufbereiten
            //--------------------------------------------------------------------------------------------------------
            var aBTComponents = tokenize( strBaseTypeMLFB );
            if ( aBTComponents.length > 1 )
            {
                // some @%$§/%%§ deliver the MLFB with some space between the last digit and the -Z for short descriptions ->
                // eliminate the resulting invalid short description 'Z'
                if ( ( aBTComponents[1] === 'Z' ) || ( aBTComponents[1] === '-Z' ) )
                    aBTComponents = removeElementsFromArray( aBTComponents, 1, 1 );
            }
            
            strBaseTypeMLFB = aBTComponents[0];
            
            if ( strBaseTypeMLFB.length === 17 ) 
            {
                // eliminate the -Z 
                strBaseTypeMLFB = strBaseTypeMLFB.substring( 0, 16 );
            }
            
            strGearTypeBT_MLFB = strBaseTypeMLFB.substring( 4, 7 );
            strMotorTypeBT_MLFB = strBaseTypeMLFB.substring( 7, 10 );
            strRatioTypeBT_MLFB = strBaseTypeMLFB.substring( 14, 16 );
            strVoltageBT_MLFB = strBaseTypeMLFB.substring( 12, 13 );
        }
        
        
        if(left(strMLFB,4) == "2KJ8")
        {
            var aComponents = tokenize( strMLFB );
            if ( !aComponents.length )
            {
                oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___FREIE_EINGABE' );
                return( oParsing )
            }
            else if ( aComponents.length > 1 )
            {
                // some @%$§/%%§ deliver the MLFB with some space between the last digit and the -Z for short descriptions ->
                // eliminate the resulting invalid short description 'Z'
                if ( ( aComponents[1] === 'Z' ) || ( aComponents[1] === '-Z' ) )
                    aComponents = removeElementsFromArray( aComponents, 1, 1 );
            }
            oParsing.strBaseMLFB = aComponents[0];
            oParsing.aTokens = aComponents;
            
            var strGearTypeMLFB = oParsing.strBaseMLFB.substring( 4, 7 );
            var strMotorTypeMLFB = oParsing.strBaseMLFB.substring( 7, 10 );
            var strRatioTypeMLFB = oParsing.strBaseMLFB.substring( 14, 16 );
            var strVoltageMLFB = oParsing.strBaseMLFB.substring( 12, 13 );
            var strTempSensorMLFB = oParsing.strBaseMLFB.substring( 10, 11 );
            var strBrakeMLFB = oParsing.strBaseMLFB.substring( 11, 12 );
            var strG115PMMLFB = oParsing.strBaseMLFB.substring( 12, 13 );
            var strG115CUMLFB = oParsing.strBaseMLFB.substring( 13, 14 );
            
            oParsing = ProcessMountingPositionContextChanged(oParsing, strBaseTypeMLFB);
            aComponents = oParsing.aTokens;
            
            //Überprüfung, ob sich Getriebe, Motor oder Übersetzung im Vergleich zur Base Type MLFB unterscheiden
            
            if( (strGearTypeMLFB!=strGearTypeBT_MLFB || strMotorTypeMLFB!=strMotorTypeBT_MLFB || strVoltageMLFB!=strVoltageBT_MLFB || strRatioTypeMLFB!=strRatioTypeBT_MLFB ) && strBaseTypeMLFB != "")
            {
                ECL_SetValue("ECSI_MLFB_CAN_CHANGE_2KJ8", "TRUE");
                
                oParsing.strBaseMLFB = "2KJ8"+strGearTypeBT_MLFB+strMotorTypeBT_MLFB+strTempSensorMLFB+strBrakeMLFB+strVoltageBT_MLFB+strG115CUMLFB+strRatioTypeBT_MLFB;
                aComponents[0]=oParsing.strBaseMLFB;
                oParsing.aTokens = aComponents;
                
                
                
                oResult = MOTOX_parse( oParsing.aTokens, strFreetext, true, false, false );
                
            }
            else
            {
                if(find(strMLFB.left(16), ".") < 0)
                {
                    //Temperatur KA setzen
                    // see 2KJ8 ShortDescriptions Database
                      
                    //vollständige SIZER MLFB
                    oResult = MOTOX_parse( oParsing.aTokens, strFreetext, true, false, false );
                }
                else
                {
                    //Temperatur KA setzen
                    var iAmbientTemperature = parseInt(ECL_GetValue("ECSI_CX_AmbientTemperatureMotor"));
                    if(isNaN(iAmbientTemperature))
                        iAmbientTemperature=25;
                    
                    if(iAmbientTemperature>40)
                        ECL_SetValue("ECSI_TEMPERATURE_SD_2KJ8", "K94");
                    
                    //Unvollständige SIZER MLFB
                    oResult = MOTOX_parse( oParsing.aTokens, strFreetext, true, true, false );
                }
            }
        }
        else
        {
            oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_BASE_FORMAT' );
        }
    }
    else
    {
    
        //--------------------------------------------------------------------------------------------------------
        // with entry of FFA single parts into the parser, we may have an order number that is not build like a 
        // normal MLFB but a string with optional leading FFA: that is build only out of numbers (at most 12)
        //--------------------------------------------------------------------------------------------------------
        //--------------------------------------------------------------------------------------------------------
        // with entry of customer-specific DEMAG crane order numbers into the parser, we may have an order number
        // that is not build like a normal MLFB but a string with required leading HGL with special appearance
        //--------------------------------------------------------------------------------------------------------
        
        var strMLFBStart = strMLFB.substring( 0, 3 );
        if ( ( strMLFBStart !== 'HGL' ) && ( strMLFBStart !== 'FFA' ) && !isOnlyNumbers( strMLFB ) && !isFFAOrderNumber( strMLFB ) )
        {
            //-------------------------------------------------------------------------------------------------------
            // message from flender says: the SiemensMall generates MLFBs so that the first short description is directly after 
            // the 'Z' without blanks or '+' or whatever. We handle that by inserting a space after the 'Z' in a normalized MLFB,
            // if none is present
            //-------------------------------------------------------------------------------------------------------
            if (( strMLFB.length > 20 )	&& (find( strMLFB, "-" ) >= 0 )) // the MLFB is normalized, but still has the two '-' for beautifying in it
            {
                if ( ( strMLFB.substring( 19, 20 ) === 'Z' ) && ( strMLFB.substring( 20, 21 ) !== ' ' ) )
                    strMLFB = strMLFB.substring( 0, 20 ) + ' ' + strMLFB.substring( 20 );
                else if ( ( strMLFB.substring( 19, 21 ) === '-Z' ) && ( strMLFB.substring( 21, 22 ) !== ' ' ) )
                    strMLFB = strMLFB.substring( 0, 21 ) + ' ' + strMLFB.substring( 21 );            
            }
            else if ( strMLFB.length > 18 ) 
            {
                // Other possibility: no '-' in mlfb, real life example: 2LC91017AD000AA0ZY99
                if ( ( strMLFB.substring( 16, 17 ) === 'Z' ) && ( strMLFB.substring( 17, 18 ) !== ' ' ) )
                    strMLFB = strMLFB.substring( 0, 17 ) + ' ' + strMLFB.substring( 17 );
                else if (( strMLFB.substring( 16, 17 ) !== ' ' ) && ( strMLFB.substring( 16, 17 ) !== 'Z' ))
                    strMLFB = strMLFB.substring( 0, 16 ) + ' ' + strMLFB.substring( 16 );
            }
            
            if ( isMotoxMLFB( strMLFB ) )
            {
                //--------------------------------------------------------------------------------------------------------
                // for motox-n and simogear, we encode some other short descriptions in an additional *OPT(a) tag
                // we must decode them here, add them to the short descriptions, and remove all references to *OPT(a)
                //--------------------------------------------------------------------------------------------------------
                strFreetext = replace(strFreetext,"(a)","@");
                var oData = decodeOpt( strMLFB, strFreetext );
                if ( oData.bSuccess )
                {
                    strMLFB = oData.strMLFB;
                    strFreetext = oData.strFreetext;
                }
            }
            else if ( strFreetext !== '' )
            {		
                //--------------------------------------------------------------------------------------------------------
                // if we have free texts, we must assure that no freetext is longer than 32 digits
                //--------------------------------------------------------------------------------------------------------
                var strErrorText = '';
                var oYDescriptions = parseFreetext( strFreetext );
                var strText = '';
                for( var p in oYDescriptions )
                {
                    if ( p === 'strError' )
                        continue;
                        
                    strText = oYDescriptions[p];
                    if ( strText.lengthUtf8 > 30 )
                    {
                        if ( strErrorText !== '' )
                            strErrorText += ' ';
                        strErrorText += replace( translate( 'ERRORMSG_OPTIONS_LENGTH_TOO_BIG' ), '{0}', p );
                    }	
                }
                if ( strErrorText !== '' )
                {
                    oParsing.strError = strErrorText;
                    return( oParsing )
                }
            
                //--------------------------------------------------------------------------------------------------------
                // with the additional short description Y66, we can encode several other short descriptions in the 
                // corresponding freetext 
                // we must decode them here, add them to the short descriptions, and remove all references to Y66 and its 
                // freetext
                //--------------------------------------------------------------------------------------------------------
                var oData = Parser.decodeY66( strMLFB, oYDescriptions );
                if ( oData )
                {
                    strMLFB = oData.strMLFB;
                    strFreetext = oData.strFreetext;
                }
            }
            
            //--------------------------------------------------------------------------------------------------------
            // separate additional entrys from base MLFB and from each other
            //--------------------------------------------------------------------------------------------------------
            var aComponents = tokenize( strMLFB );
            if ( !aComponents.length )
            {
                oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___FREIE_EINGABE' );
                return( oParsing )
            }
            else if ( aComponents.length > 1 )
            {
                // some @%$§/%%§ deliver the MLFB with some space between the last digit and the -Z for short descriptions ->
                // eliminate the resulting invalid short description 'Z'
                if ( ( aComponents[1] === 'Z' ) || ( aComponents[1] === '-Z' ) )
                    aComponents = removeElementsFromArray( aComponents, 1, 1 );
            }
            oParsing.strBaseMLFB = aComponents[0];
        
            // normally, normalized MLFBs have 16 or 17 digits (the latter with "-Z" at  the } to ancounce additional entrys)
            if ( ( oParsing.strBaseMLFB.length !== 16 ) && ( oParsing.strBaseMLFB.length !== 17 ) )
            {
                oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_WRONG_SIZE' );
                return( oParsing );
            }
        
            var bWantShortDescription = false;
            if ( oParsing.strBaseMLFB.length === 17 ) 
            {
                // eliminate the -Z 
                oParsing.strBaseMLFB = oParsing.strBaseMLFB.substring( 0, 16 );
                aComponents[0] = oParsing.strBaseMLFB;
                bWantShortDescription = true;
            }
        
            // if we marked with a '-Z' that we have short descriptions, we must have them as part of the order number
            if ( bWantShortDescription && ( aComponents.length <= 1 ) )
            {			
                if ( bIsCouplingMLFB )
                    oParsing.strError = translate( 'ERRORMSG_COUPLINGMLFB_FEHLER_BEI_MLFB_ANALYSE' );
                else oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_NO_SHORT_DESCRIPTIONS' );
                return( oParsing );
            }
            
            //--------------------------------------------------------------------------------------------------------
            // do some syntactical checks for the additional entrys
            //--------------------------------------------------------------------------------------------------------
            var i = 1;
            var strSD = '', strSDCollector = '';
            while ( i < aComponents.length )
            {
                strSD = aComponents[i];
                
                // check KA size
                if ( ( strSD.length > 3 ) && ( findex( strSD, '\"', 3, 1 ) < 0 ) && ( findex( strSD, 'Y', 1, 1 ) < 0 ) )
                {
                    if ( bIsCouplingMLFB )
                        oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_WRONG' );
                    else oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_WRONG_SIZE_KURZANGABE' ) + ' ' + strSD;
                    return( oParsing );
                }
                else if ( strSD.length < 3 )
                {
                    oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_WRONG_SIZE_KURZANGABE' ) + ' ' + strSD;
                    return( oParsing );
                }
                        
                // check if ka has been processed before
                strSD = strSD.substring( 0, 4 );
                if ( strSDCollector.indexOf( strSD ) > -1 )
                {
                    
                    oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_DOPPELTE_KURZANGABE' ) + ' ' + strSD;
                    return( oParsing );
                }
                strSDCollector += strSD;
                i++;
            }
            oParsing.aTokens = aComponents;
        }
        else
        {
            oParsing.aTokens = [ strMLFB ];
            oParsing.strBaseMLFB = strMLFB;
            // here we have no freetext to account for
        }
            
        //--------------------------------------------------------------------------------------------------------
        // now we spread the parsing to the product-specific parsing functions
        //--------------------------------------------------------------------------------------------------------
        
        var strMLFBStart = oParsing.strBaseMLFB.substring( 0, 3 );
        if ( isSimoGearMLFB( oParsing.strBaseMLFB ) )
        {
            // '2KJ8' which means SimoGear with G115D
            oResult = MOTOX_parse( oParsing.aTokens, strFreetext, bnotfromProjectList, false, false );
        }
        else
        {
            oParsing.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_BASE_FORMAT' );
        }
    }	
    // assemble results
    if ( oResult )
    {
        if ( oResult.oProduct )
            oParsing.oProduct = oResult.oProduct;
        if ( oResult.oParsingData )
            oParsing.oParsingData = oResult.oParsingData;
        if ( oResult.oShortDescriptions )
            oParsing.oShortDescriptions = oResult.oShortDescriptions;
        if ( oResult.strModifiedMLFB )	//may have been modified by some compatibility issues
            oParsing.strBaseMLFB = oResult.strModifiedMLFB;
        oParsing.strError += ( ( oParsing.strError ? ' ' : '' ) + oResult.strError );
        oParsing.strWarning += ( ( oParsing.strWarning ? ' ' : '' ) + oResult.strWarning );
        
        //Sep. 2014 special for flenderprojecthandler region
        oParsing.strRegion = ( ( oResult.strRegion ? oResult.strRegion : '' ));
    }
    
    vtTimeStop	= Date.getTime();
    vtTime		= (vtTimeStop - vtTimeStart) / 1000;
    logtraceln("******* Parse - Time: "+vtTime+" sec"+"\n");

    return( oParsing );
}

function _checkBaseMLFB( strBaseMLFB, oShortDescriptions, strRegion, incompleteSizerMLFB, KeepConfiguration )
{
    var oResult = {
        strError: '',
        strWarning: '',
        oParsingData: {},
    };
    
    // -------------------------------------------------------------------------------------------------
    // the fourth digit gives us the 'Generationenziffer'  (Numeric)
    var strGenerationDigitMLFB = strBaseMLFB.charAt( 3 );
    if ( !checkGenerationDigit( strGenerationDigitMLFB ) )
    {
        // that must not happen
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____GENERATION_DIGIT' );
        return( oResult );
    }

    // digits 5-7: gear type
    var strGearTypeMLFB = strBaseMLFB.substring( 4, 7 );
    var oGearType = checkGearType( strGearTypeMLFB, strRegion );
    if ( !oGearType )
    {
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____GEAR_TYPE' );
        return( oResult );
    }
    
    if(oGearType.EGBEZBRA == "K")
    {
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____GEAR_COOLING_TOWER' );
        return( oResult );
    }
    
    oResult.oParsingData.oGearType = oGearType;
    
    // digits 15+16: gear ratio
    var strRatioMLFB = strBaseMLFB.substring( 14, 16 );
    var oRatio = checkRatio( strRatioMLFB, oGearType );
    if ( !oRatio )
    {
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____RATIO' );
        return( oResult );
    }
    oResult.oParsingData.oRatio = oRatio;
    
    // digits 8-10: motor size, power, type and voltage
    //Antriebsgruppe
    var strMotorMLFB = strBaseMLFB.substring( 7, 10 );
    var strG115PMMLFB = strBaseMLFB.charAt( 12 );
    
    //Motor
    oResult.oParsingData.strProduct = "GETRIEBEMOTOR";
    
    var oMotor = checkMotor( strMotorMLFB, strG115PMMLFB, oGearType, oShortDescriptions, strRegion , incompleteSizerMLFB );
    if ( !oMotor )
    {
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____WRONG_MOTOR' );
        return( oResult );
    }
    oResult.oParsingData.oMotor = oMotor;
    
    //Motorschutz, nur PT1000
    var strMotorProtectionMLFB = strBaseMLFB.charAt( 10 );
    if(strMotorProtectionMLFB != ".")
    {
        if(strMotorProtectionMLFB == "1")
            oResult.oParsingData.MotorProtection = "Pt1000";
        else
            oResult.oParsingData.MotorProtection = "ohne";
    }
    
    //Bremse nur mit/ohne
    var strBrakeMLFB = strBaseMLFB.charAt( 11 );
    if(strBrakeMLFB != ".")
    {
        if(strBrakeMLFB != "0")
       {
            var oBrake = checkBrake(strBrakeMLFB, oGearType, oMotor, oShortDescriptions, strRegion);
            if ( !oBrake )
            {
                oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_G115D' );
                return( oResult );
            }
            else
            {
                oResult.oParsingData.Brake = "mit";
                oResult.oParsingData.oBrake = oBrake;
            }
       }
        else	
            oResult.oParsingData.Brake = "ohne";
    }
    
    //G115D MLFB
    var strG115DMLFB = strBaseMLFB.charAt( 13 );
    if(strG115DMLFB!=".")
    {
        var oG115D = checkG115D( strG115DMLFB, oGearType, oMotor, oShortDescriptions, strRegion);
        if ( !oG115D )
        {
            oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____WRONG_G115D' );
            return( oResult );
        }
        oResult.oParsingData.oG115D = oG115D;
    }

    //Unterscheidung Wall oder Motormounted
    if(oResult.oParsingData.oMotor.G115M == "1")
        oResult.oParsingData.strG115 = "MM";
    else if(oResult.oParsingData.oMotor.G115D == "1")
        oResult.oParsingData.strG115 = "WM";
    
    var bRet = checkMotorGetriebeBaubarkeit( oResult, strRegion );
    var bRet2 = checkMotorGetriebeBaubarkeit2( oResult, strRegion );
	if(!bRet || !bRet2)
    {
        oResult.strError = translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____GEARMOTORCHECK' );
        return( oResult );
    }

    
    return( oResult );
}


/**
 * @constructor
 * helper parsing routine for SimoGear geared motors that performs some checking for the neccessary short descriptions of the base MLFB
 *
 * @param oShortDescriptions - object for short descriptions (and optional corresponding freetext)
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group and freetext (the freetext string may be empty)
 *
 * @return string - error message
 *                  empty all freetexts are present and have the right format
 */
function _checkFreetexts( oShortDescriptions )
{
    // test on missing freetexts
    var strMissingFreetext = '';
    if ( ( 'Y00' in oShortDescriptions ) && ( oShortDescriptions.Y00.freetext === '' ) )
        strMissingFreetext += 'Y00, ';
    if ( ( 'Y80' in oShortDescriptions ) && ( oShortDescriptions.Y80.freetext === '' ) )
        strMissingFreetext += 'Y80, ';
    if ( strMissingFreetext !== '' )
        return( translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_MISSING_TEXT_DESCRIPTION' ) + ' (' + trimright( strMissingFreetext, ', ' ) + ')' );
    
    // test if Y80 - coloring is possible -> is now done in _checkGearedMotorOptions
    
    return( '' );
}


/**
 * @constructor
 * helper parsing routine for SimoGear geared motors that performs some checking if the base gear options match together, i.e. the geared motor is buildable
 *
 * @param oSimulatedGearedMotor - geared motor object
 * @param oParsingData - results from the parsing process
 *                       oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *									EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                       oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                       oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                       oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                       oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *                  			 MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                               WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, MOTSPG, SUCHTYP etc. (all columns of table CD_MOTOREN)
 *
 * @return string - error message
 *                  empty if the geared motor is buildable
 */
function _checkGearedMotor( oSimulatedGearedMotor, oParsingData )
{
    // this will be the right place to check if the base gear can be assembled to the base motor 
    
    return( '' );
}


/**
 * @constructor
 * helper parsing routine for SimoGear geared motors that performs some checking if the configuration options noted by short descriptions  match together, 
 * i.e. the geared motor is buildable
 *
 * @param oSimulatedGearedMotor - geared motor object
 * @param oShortDescriptions - object for short descriptions (and optional corresponding freetext)
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group and freetext (the freetext string may be empty)
 * @param oParsingData - results from the parsing process
 *                       oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *									EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                       oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                       oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                       oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                       oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *                  			 MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                               WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, MOTSPG, SUCHTYP etc. (all columns of table CD_MOTOREN)
 *
 * @return string - error message
 *                  empty if the gear is buildable
 */
function _checkGearedMotorOptions( oSimulatedGearedMotor, oShortDescriptions, oParsingData, strRegion )
{
    //---------------------------------------------------------------------------------------------------------------------
    // motor
    //---------------------------------------------------------------------------------------------------------------------

    //der Anschlusskasten vergossen ist bei CSA / UL-R nicht möglich
    //Anschlusskasten M53
    //CSA /UL-R N38	
    if('N38' in oShortDescriptions && 'M53' in oShortDescriptions)
        return( translate( 'PARSING_ERROR_2KJ___M53_N38' ) );

    //Spezifikationsstring übertragen
    oSimulatedGearedMotor.strSpecification = oParsingData.oMotor.MOTELAUSF_AUSG;
    
    // test if venting is possible for this motor
    var aVentings = Product.MOTOX.Config.MechanicalDesign.getCombo_Belueftung( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
    if ( !Base.hasConfigurationOption( aVentings, oSimulatedGearedMotor.oConfig.strMechanicalVenting ) )
    {
        var strVentingSD = Parser.getSDFromGroup( oShortDescriptions, 'MOTORBELUEFTUNG' );
        if ( strVentingSD !== '' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_VENTING' ) + ' (' + strVentingSD + ')' );
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_VENTING' ) );
    }
    
    // test that motor connector, mating and emc-shielding match
    var strSDConnector = Parser.getSDFromGroup( oShortDescriptions, 'MOTORSTECKER' );
    var strSDMating = Parser.getSDFromGroup( oShortDescriptions, 'MOTORGEGENSTECKER' );
    if ( ( strSDConnector === '' ) && ( strSDMating !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_MATING_WITHOUT_MOTOR_CONNECTOR' ) );
    if ( strSDConnector !== '' )
    {
        var aConnectors = Product.MOTOX.Config.MechanicalDesign.getCombo_Motorstecker( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion}  );
        if ( !Base.hasConfigurationOption( aConnectors, oSimulatedGearedMotor.oConfig.strMotorConnector ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____WRONG_MOTOR_CONNECTOR' ) );
        if ( oSimulatedGearedMotor.oConfig.bEMCshielded && !Product.MOTOX.Config.MechanicalDesign.getCheckbox_Motorstecker_EMV( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_EMC_SHIELDED' ) );
        var aMatings = Product.MOTOX.Config.MechanicalDesign.getCombo_Gegenstecker( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor );
        if ( !Base.hasConfigurationOption( aMatings, oSimulatedGearedMotor.oConfig.strMating ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____WRONG_MOTOR_CONNECTOR_MATING' ) );		
    }

    //---------------------------------------------------------------------------------------------------------------------
    // brake
    //---------------------------------------------------------------------------------------------------------------------
    var strBrakeSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_TYP' );
    var strBrakeVoltageSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_SPANNUNG' );
    var strBrakeFunctionRectifierSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_FUNKTIONSGLEICHRICHTER' );
    var strBrakeRectifierSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_GLEICHRICHTER' );
    var strBrakeControlSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_SCHALTUNG' );
    var strBrakeDesignSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_AUSFUEHRUNG' );
    var strBrakeCorrosionProtectionSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_KORROSIONSSCHUTZ' );
    var strBrakeWearSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_BELAG' );
    var strBrakeManualReleaseSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_HANDLUEFTUNG' );
    var strBrakeManualReleaseLocationSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_HANDLUEFTUNG_LAGE' );
    var strBrakeMonitorSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_MONITOR' );
    var strBrakeCableProtectionSD = Parser.getSDFromGroup( oShortDescriptions, 'BREMSE_KABELSCHUTZ' );
    
    // one cannot have any brake attribute without a brake type
    if ( ( strBrakeSD === '' ) && 
         ( ( strBrakeVoltageSD !== '' ) || ( strBrakeFunctionRectifierSD !== '' ) || ( strBrakeRectifierSD !== '' ) || ( strBrakeControlSD !== '' ) || ( strBrakeDesignSD !== '' ) || ( strBrakeCorrosionProtectionSD !== '' ) ||
           ( strBrakeWearSD !== '' ) || ( strBrakeManualReleaseSD !== '' ) || ( strBrakeManualReleaseLocationSD !== '' ) || ( strBrakeMonitorSD !== '' ) || ( strBrakeCableProtectionSD !== '' ) ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_BRAKE' ) );
        
    // test if brake type is possible on this motor
    if ( strBrakeSD !== '' )
    {
        if ( !Product.MOTOX.Config.Brake.isBremsemoeglich( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_NOT_POSSIBLE' ) );
            
        var aBrakes = Product.MOTOX.Config.Brake.getTable_generateValidBrakes( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
        if ( !Base.hasConfigurationOption( aBrakes, oSimulatedGearedMotor.oConfig.strSelBrake ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_NOT_POSSIBLE_FOR_THIS_MOTOR' ) + ' (' + strBrakeSD + ')' );
            
        // function rectifier and simple recifier are not possible together
        if ( ( strBrakeFunctionRectifierSD !== '' ) && ( strBrakeRectifierSD !== '' ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_DOUBLE_RECTIFIER' ) );
    }
        
    // brake on terminal board will only be possible with certain function rectifiers
    if ( strBrakeControlSD === 'C20' )
    {
        if ( oSimulatedGearedMotor.oConfig.strBrakeFunctionRectifier.indexOf( 'Stromerf.' ) < 0 && (oSimulatedGearedMotor.strProduct != "GETRIEBEMOTOR.G110M" && oSimulatedGearedMotor.strProduct != "GETRIEBEMOTOR.G110M_SIP"))
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_ON_TERMINAL_BOARD' ) );
    }
    
    // if a manual brake release location is given, we also must have a brake manual release
    if ( ( strBrakeManualReleaseLocationSD !== '' ) && ( strBrakeManualReleaseSD === '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_NO_MANUAL_RELEASE' ) );
    if ( ( strBrakeManualReleaseLocationSD === '' ) && ( strBrakeManualReleaseSD !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BRAKE_NO_MANUAL_RELEASE_LOCATION' ) );
    
    // ... to enhance for yet unknown problems
    
    //---------------------------------------------------------------------------------------------------------------------
    // gear
    //---------------------------------------------------------------------------------------------------------------------
    // if we have a short description for torque support figur, the fastening must be a torque support-type
    var strTorqueSupportFigurSD = Parser.getSDFromGroup( oShortDescriptions, 'GETRIEBE_FIGUR_DREHMOMENTSTUETZE' );
    if ( ( strTorqueSupportFigurSD !== '' ) && ( oSimulatedGearedMotor.oConfig.strMountingType !== 'D' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_TORQUE_SUPPORT_FIGUR_WITHOUT_TORQUE_SUPPORT' ) + ' (' + strTorqueSupportFigurSD + ')' );
    // if we have a short description for flange diameter, the fastening must be a flange-type
    var strFlangeDiameterSD = Parser.getSDFromGroup( oShortDescriptions, 'FLANSCHDURCHMESSER' );
    if ( strFlangeDiameterSD !== '' )
    {
        var aDiameters = Product.MOTOX.Config.Design.getCombo_Befestigungsmass( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
        if ( !Base.hasConfigurationOption( aDiameters, oSimulatedGearedMotor.oConfig.strFittingDimension ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____FLANGE_DIAMETER_NOT_POSSIBLE' ) + ' (' + strFlangeDiameterSD + ')' );
    }
    
    // backstop tests
    var strGearBackstopSD = Parser.getSDFromGroup( oShortDescriptions, 'GETRIEBE_RUECKLAUFSPERRE' );
    var strMotorBackstopSD = Parser.getSDFromGroup( oShortDescriptions, 'MOTOR_RUECKLAUFSPERRE' );
    var strDirectionSD = Parser.getSDFromGroup( oShortDescriptions, 'GETRIEBE_DREHRICHTUNG_ABTRIEBSWELLE' );
    var strDirectionPresentSD = Parser.getSDFromGroup( oShortDescriptions, 'GETRIEBE_DREHRICHTUNG_VORGEGEBEN' );
    // if a short description for any backstop is present, we must also have a short description for rotation direction (and no K79)
    // -og- 120420 with the new handling of rotating direction, we may have a backstop without short description for rotating direction
    //             in this case, a default rotating direction is calculated
    //if ( ( ( strGearBackstopSD !== '' ) || ( strMotorBackstopSD !== '' ) ) && ( strDirectionSD === '' ) )
    //	return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BACKSTOP_NO_DIRECTION' ) );
    // if a rotating direction is announced, a rotating direction must be present
    
    if ( ( strDirectionSD === '' ) && ( strDirectionPresentSD !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____DIRECTION_NOT_ANNOUNCED' ) );
    // if we have a rotation direction, we must either have a backstop or it must be announced
    
    //K79 is no longer activ
    /*
    if ( ( strDirectionSD !== '' ) && ( strGearBackstopSD === '' ) && ( strMotorBackstopSD === '' ) && ( strDirectionPresentSD === '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____DIRECTION_PRESENT' ) );
    */
    // if we have a backstop, is rotation direction must not be announced
    if ( ( ( strGearBackstopSD !== '' ) || ( strMotorBackstopSD !== '' ) ) && ( strDirectionPresentSD !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BACKSTOP_DIRECTION_NOT_ANNOUNCED' ) );
    
    //SELECT DISTINCT * FROM MOTYPX WHERE M3 = '1' AND AKTIV = '1' AND CDROM_REL = '1' AND BELSDACH = '-' AND BELART = '-' AND SHWG = '-' AND MOTRLSP = '-' AND MOTYPX.HEMOTOR = '-' AND MOTYPX.HANDRAD = '-' 
    //Überprüfung, ob die Motoranbauten in Kombination zusammenpassen
    var strDBName = "SIMOGEAR_DB";
    var ismoeglich = false;
    var strDBResult = "MOTYPX_TEST";
    
    var bHandrad = Parser.getSDFromGroup( oShortDescriptions, 'MOTOR_HANDRAD' );
    var bSchutzdach = Parser.getSDFromGroup( oShortDescriptions, 'MOTOR_SCHUTZDACH' );
    var bRLSP = Parser.getSDFromGroup( oShortDescriptions, 'MOTOR_RUECKLAUFSPERRE' );
    var b2tesWellenende = Parser.getSDFromGroup( oShortDescriptions, 'MOTOR_ZWEITESWELLENENDE' );
    var strBelueftung = Parser.getSDFromGroup( oShortDescriptions, 'MOTORBELUEFTUNG' );
    
    var strSelect = "SELECT DISTINCT * FROM MOTYPX WHERE ";
    strSelect = strSelect + "(AKTIV = '1' OR AKTIV = '2') AND ";
    strSelect = strSelect + "M3 = '"+oParsingData.oMotor.M3+"' AND ";
    strSelect = strSelect + "GM3 = '"+oParsingData.oMotor.GM3+"' AND ";
    strSelect = strSelect + "CDROM_REL = '1' AND ";
    strSelect = strSelect + "MOTYPX.S09 = '0' AND ";
    
    if(bSchutzdach)
        strSelect = strSelect + "BELSDACH = 'W' AND ";
    else
        strSelect = strSelect + "BELSDACH = '-' AND ";
    
    if(strBelueftung == "F")
        strSelect = strSelect + "BELART = 'F' AND ";
    else if(strBelueftung == "U")
        strSelect = strSelect + "BELART = 'U' AND ";
    else
        strSelect = strSelect + "BELART = '-' AND ";
    
    if(strBelueftung == "SHWG")
        strSelect = strSelect + "SHWG = 'I' AND ";
    else
        strSelect = strSelect + "SHWG = '-' AND ";

    if(oParsingData.strProduct == "GETRIEBEMOTOR.G110M")
        strSelect = strSelect + "G110M = 'M' AND ";
    else
        strSelect = strSelect + "G110M = '-' AND ";
    
    if(bRLSP)
        strSelect = strSelect + "MOTRLSP = 'X' AND ";
    else
        strSelect = strSelect + "MOTRLSP = '-' AND ";
    
    var strHEMOTOR="-";
    var strMotbez=oParsingData.oMotor.SUCHTYP;	
    strMotbez=trimright(strMotbez,"V");
    if(strMotbez.right(1)=="S")
        strHEMOTOR="S"
    else if(strMotbez.right(1) == "E")
        strHEMOTOR = 'E';
    else if(strMotbez.right(1) == "P")
        strHEMOTOR = 'P';
    
    strSelect = strSelect + "MOTYPX.HEMOTOR = '"+strHEMOTOR+"' AND ";
    
    if(bHandrad)
        strSelect = strSelect + "MOTYPX.HANDRAD = 'D' AND ";
    else
        strSelect = strSelect + "MOTYPX.HANDRAD = '-' AND ";
    
    strSelect = strSelect + "MOTYPX.VSDMOTOR = '"+oParsingData.oMotor.VSDMOTOR+"' AND ";
    
    if(strRegion == "CHINA")
        strSelect = strSelect + "(MOTYPX.REGION = 'C' OR MOTYPX.REGION = 'SC' OR MOTYPX.REGION = 'SCM' OR MOTYPX.REGION = 'SMC' OR MOTYPX.REGION = 'SMCI') ";
    else
        strSelect = strSelect + "(MOTYPX.REGION = 'S' OR MOTYPX.REGION = 'SC' OR MOTYPX.REGION = 'SCM' OR MOTYPX.REGION = 'SM' OR MOTYPX.REGION = 'SMC' OR MOTYPX.REGION = 'SMCI') ";
        
    if ( dbselect( strDBName, strSelect, strDBResult ) > 0 )
    {
        ismoeglich = true;
    }
    else
    {
        ismoeglich = false;
    }
    
    if(bHandrad && !b2tesWellenende)
        ismoeglich = false;
        
    if(bSchutzdach && b2tesWellenende)
        ismoeglich = false;
            
    if(!ismoeglich)
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____MOTORANBAUTEN' ) );
    
    // short descriptions for 'hollow shaft cover' are only possible if the output shaft is of hollow shaft type
    var strHollowShaftCoverSD = Parser.getSDFromGroup( oShortDescriptions, 'GETRIEBE_ABDECKUNG_HOHLWELLE' );
    if ( strHollowShaftCoverSD !== '' &&  strHollowShaftCoverSD !== "G58")
    {
        var strShaftType = oSimulatedGearedMotor.oConfig.strOutputShaftDesign.charAt( 0 );
        if ( strShaftType !== 'A' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____HOLLOW_SHAFT_COVER' ) );
    }
    // ... to enhance for yet unknown problems
    
    //---------------------------------------------------------------------------------------------------------------------
    // conservation
    //---------------------------------------------------------------------------------------------------------------------
    // coloring for lacquer is only possible if we have a lacquer type of LACKIERT
    var strColorSD = Parser.getSDFromGroup( oShortDescriptions, 'KONSERVIERUNG_FARBTON' );
    if ( ( strColorSD !== '' ) && ( oSimulatedGearedMotor.oConfig.strConservationPainting !== 'LACKIERT' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_COLOR_FOR_LACQUER' ) );
            
    // additional lacquer on flange is only possible with a flange-type fastening
    var strLacquerOnFlangeSD = Parser.getSDFromGroup( oShortDescriptions, 'KONSERVIERUNG_FLANSCH' );
    if ( ( strLacquerOnFlangeSD === 'L11' ) || ( strLacquerOnFlangeSD === 'L12' ) )
    {
        if ( oSimulatedGearedMotor.oConfig.strConservationPainting === 'UNLACKIERT' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE' ) );
        if ( oSimulatedGearedMotor.oConfig.strMountingType !== 'F' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE_WITHOUT_FLANGE' ) );
    }
    else if ( strLacquerOnFlangeSD === 'L13' )
    {
        if ( ( oSimulatedGearedMotor.oConfig.strMountingType !== 'Z' ) || ( ( ( oSimulatedGearedMotor.GearData.EGBEZTYP !== 'K' ) && ( oSimulatedGearedMotor.GearData.EGBEZTYP !== 'B' ) ) ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE_WITHOUT_FLANGE' ) );
    }
    
    // check if special coloring is possible
    if ( 'Y80' in oShortDescriptions )
    {
        var strColorID = right(oSimulatedGearedMotor.oConfig.strConservationColor,4);
        var aColors = Product.MOTOX.Config.Conservation.getCombo_Farbe( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
        if ( !Base.hasConfigurationOption( aColors, strColorID ) )
            return( translate( 'ERRORMSG_SONDERFARBTON' ) );
    }
    
    
    //Check, ob Motorspannung, Motorstecker und Funktionsgleichrichter passen
    var bRet = Parser.MOTOX.checkSpannungMotorsteckerFktglr(oSimulatedGearedMotor.oConfig,oSimulatedGearedMotor, oSimulatedGearedMotor,strRegion);
    if(!bRet)
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____SPANNUNG_MOTORSTECKER_FKTGLR' ) );
    
    
    // ... to enhance for yet unknown problems
    
    return( '' );
}

function _checkGearedAdapterOptions( oSimulatedGearedMotor, oShortDescriptions, oParsingData, strRegion )
{
    //---------------------------------------------------------------------------------------------------------------------
    // gear
    //---------------------------------------------------------------------------------------------------------------------
    // if we have a short description for torque support figur, the fastening must be a torque support-type
    var strTorqueSupportFigurSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_FIGUR_DREHMOMENTSTUETZE' );
    if ( ( strTorqueSupportFigurSD !== '' ) && ( oSimulatedGearedMotor.oConfig.strMountingType !== 'D' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_TORQUE_SUPPORT_FIGUR_WITHOUT_TORQUE_SUPPORT' ) + ' (' + strTorqueSupportFigurSD + ')' );
    // if we have a short description for flange diameter, the fastening must be a flange-type
    var strFlangeDiameterSD = getSDFromGroup( oShortDescriptions, 'FLANSCHDURCHMESSER' );
    if ( strFlangeDiameterSD !== '' )
    {
        var aDiameters = Product.MOTOX.Config.Design.getCombo_Befestigungsmass( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
        if ( !Base.hasConfigurationOption( aDiameters, oSimulatedGearedMotor.oConfig.strFittingDimension ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____FLANGE_DIAMETER_NOT_POSSIBLE' ) + ' (' + strFlangeDiameterSD + ')' );
    }
    
    // backstop tests
    var strGearBackstopSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_RUECKLAUFSPERRE' );
    var strAdapterBackstopSD = getSDFromGroup( oShortDescriptions, 'AV_RLSP' );
    var strDirectionSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_DREHRICHTUNG_ABTRIEBSWELLE' );
    var strDirectionPresentSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_DREHRICHTUNG_VORGEGEBEN' );
    // if a short description for any backstop is present, we must also have a short description for rotation direction (and no K79)
    // -og- 120420 with the new handling of rotating direction, we may have a backstop without short description for rotating direction
    //             in this case, a default rotating direction is calculated
    //if ( ( ( strGearBackstopSD !== '' ) || ( strMotorBackstopSD !== '' ) ) && ( strDirectionSD === '' ) )
    //	return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BACKSTOP_NO_DIRECTION' ) );
    // if a rotating direction is announced, a rotating direction must be present
    
    if ( ( strDirectionSD === '' ) && ( strDirectionPresentSD !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____DIRECTION_NOT_ANNOUNCED' ) );
    // if we have a rotation direction, we must either have a backstop or it must be announced
    
    //K79 is no longer activ
    /*
    if ( ( strDirectionSD !== '' ) && ( strGearBackstopSD === '' ) && ( strAdapterBackstopSD === '' ) && ( strDirectionPresentSD === '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____DIRECTION_PRESENT' ) );
    */
    // if we have a backstop, is rotation direction must not be announced
    if ( ( ( strGearBackstopSD !== '' ) || ( strAdapterBackstopSD !== '' ) ) && ( strDirectionPresentSD !== '' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____BACKSTOP_DIRECTION_NOT_ANNOUNCED' ) );
    
    // short descriptions for 'hollow shaft cover' are only possible if the output shaft is of hollow shaft type
    var strHollowShaftCoverSD = getSDFromGroup( oShortDescriptions, 'GETRIEBE_ABDECKUNG_HOHLWELLE' );
    if ( strHollowShaftCoverSD !== '' &&  strHollowShaftCoverSD !== "G58")
    {
        var strShaftType = oSimulatedGearedMotor.oConfig.strOutputShaftDesign.charAt( 0 );
        if ( strShaftType !== 'A' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____HOLLOW_SHAFT_COVER' ) );
    }
    // ... to enhance for yet unknown problems
    
    //---------------------------------------------------------------------------------------------------------------------
    // conservation
    //---------------------------------------------------------------------------------------------------------------------
    // coloring for lacquer is only possible if we have a lacquer type of LACKIERT
    var strColorSD = getSDFromGroup( oShortDescriptions, 'KONSERVIERUNG_FARBTON' );
    if ( ( strColorSD !== '' ) && ( oSimulatedGearedMotor.oConfig.strConservationPainting !== 'LACKIERT' ) )
        return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____NO_COLOR_FOR_LACQUER' ) );
            
    // additional lacquer on flange is only possible with a flange-type fastening
    var strLacquerOnFlangeSD = getSDFromGroup( oShortDescriptions, 'KONSERVIERUNG_FLANSCH' );
    if ( ( strLacquerOnFlangeSD === 'L11' ) || ( strLacquerOnFlangeSD === 'L12' ) )
    {
        if ( oSimulatedGearedMotor.oConfig.strConservationPainting === 'UNLACKIERT' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE' ) );
        if ( oSimulatedGearedMotor.oConfig.strMountingType !== 'F' )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE_WITHOUT_FLANGE' ) );
    }
    else if ( strLacquerOnFlangeSD === 'L13' )
    {
        if ( ( oSimulatedGearedMotor.oConfig.strMountingType !== 'Z' ) || ( ( ( oSimulatedGearedMotor.GearData.EGBEZTYP !== 'K' ) && ( oSimulatedGearedMotor.GearData.EGBEZTYP !== 'B' ) ) ) )
            return( translate( 'ERRORMSG_PARSER_SIMOGEAR_2KJ____LACQUER_ON_FLANGE_WITHOUT_FLANGE' ) );
    }
    
    // check if special coloring is possible
    if ( 'Y80' in oShortDescriptions )
    {
        var strColorID = oSimulatedGearedMotor.oConfig.strConservationColor;
        var aColors = Product.MOTOX.Config.Conservation.getCombo_Farbe( oSimulatedGearedMotor.oConfig, oSimulatedGearedMotor, { strAmbient: 'NORMAL' ,strRegion : strRegion} );
        if ( !Base.hasConfigurationOption( aColors, strColorID ) )
            return( translate( 'ERRORMSG_SONDERFARBTON' ) );
    }
    
    // ... to enhance for yet unknown problems
    
    return( '' );
}


/**
 * @constructor
 * main parsing routine for MOTOX geared motors
 *
 * @param aTokens - array with MLFB tokens; the first array entry is guaranteed to be the base MLFB
 *                  all others denote exaclty one short description (without additional free text)
 * @param strFreetext - encoded freetext for short descriptions
 * @param bnotfromProjectList - signals if the MLFB is parsed from direct input/SAP converter or from project list
 *
 * @return object - results from the parsing process
 *                 strError - error message from parsing (empty, if parsing succeeded)
 *                 strWarning - additional warning message (may be nonempty even if parsing succeeded)
 *                 oShortDescriptions - object for short descriptions (and optional corresponding freetext)
 *                                      each attribute name is a short description
 *                                      each attribute value is an object with attributes group and freetext (the freetext string may be empty)
 *                                      may not be present on parsing error
 *                 oProduct - MOTOX product object (will not be present on parsing errors)
 *                 oParsingData - parsing data object: contains parsing data results from different parsing stages (not all subobjects may be present due to parsing errors)
 *                                oGearType: parsing results for MLFB gear type part, contains attributes MLFB_2KJ_5, MLFB_2KJ_6, MLFB_2KJ_7, EGBEZTYP, EGBEZUEBS, EGBEZDIM, 
 *											 EGBEZEI, ZGBEZUEBS, ZGBEZDIM, ZGBEZEI
 *                                oShaft: parsing results for MLFB gear output shaft part, contains attributes EGTYP, EGABW, EGBEZAWTYP, EGBEZWNV, EGBEZBRA, ANWTYP, EGBEZSO
 *                                oFastening: parsing results for MLFB gear fastening part, contains attributes MLFB_2KJ_14, MLFB_2KJ_14_KA, EGBEZBEF
 *                                oRatio: parsing results for MLFB gear ratio part, contains attributes DGUEB, ZGUEB, EGUEB, ETA, T2FB1, N2
 *                                oMotor: parsing results for MLFB motor part, contains attributes MLFB_2KJ_9, MLFB_2KJ_10, MLFB_2KJ_11, MLFB_2KJ_12, MLFB_2KJ_13, MLFB_2KJ_P_KA, 
 *                  					  MLFB_2KJ_13_KA, SO_NEMA_KA, SO_NEMA, SO_CSA_UL_R_KA, SO_CSA_UL_R, SO_GOST_KA, SO_GOST, SO_CHINA_KA, SO_CHINA, SO_CE_KA, SO_CE, WARMEKL, 
 *                                        WIBLNR, WIBLAUSLG, POLZAHL, FREQUENZ, MOTDIM, FU, NETZ, MOTSPG, SUCHTYP etc. (all columns of table CD_MOTOREN)
 */
function MOTOX_parse( aTokens, strFreetext, bnotfromProjectList, incompleteSizerMLFB, KeepConfiguration )
{
    var oResult = {
        strError: '',
        strRegion: '',
        strWarning: '',
        oProduct: null,
        oParsingData: null,
        oShortDescriptions: null
    };
    
    var strBaseMLFB = "";
    if(aTokens !== undefined){
        strBaseMLFB = aTokens[0];
    }
    
    if ( isMotoxNMLFB( strBaseMLFB ) )
    {
        // Motox-N cannot be parsed by this function. You must have caught MOTOX-N geared motors before and submitted them to some conversion routine
        oResult.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_NO_MOTOXN' );
        return( oResult );
    }
    
    
    //---------------------------------------------------------------------------------------------------------------------
    // freetext analysis
    // since we need some information for necessary additions while checking the base MLFB, we must build the 
    // necessary structures before analysing the base MLFB
    //---------------------------------------------------------------------------------------------------------------------
    var oShortDescriptions = buildShortDescriptions( aTokens, strFreetext, '2KJ8_KURZANGABEN_DB' );	
    if ( oShortDescriptions.strError )
    {
        oResult.strError = oShortDescriptions.strError;
        return( oResult );
    }
    if ( oShortDescriptions.strWarning )
        oResult.strWarning = oShortDescriptions.strWarning;
    oResult.oShortDescriptions = oShortDescriptions;
    
    var strRegion;
    if(bnotfromProjectList)
    {
        strRegion = "TUB";
    }
    else
    {
        strRegion = "TUB";
    }
    
    oResult.strRegion = strRegion;

    //---------------------------------------------------------------------------------------------------------------------
    // now, we do some syntactical checking on the MLFB 
    // if this function is called, we know for sure that the first four digits are '2KJ3'
    //---------------------------------------------------------------------------------------------------------------------
    var oBase = _checkBaseMLFB( strBaseMLFB, oShortDescriptions, strRegion, incompleteSizerMLFB, KeepConfiguration );
    oResult.strError = oBase.strError;
    oResult.oParsingData = oBase.oParsingData;
    if ( oResult.strError !== '' )
        return( oResult );
    
    // check freetexts
    strError = _checkFreetexts( oShortDescriptions );
    if ( strError !== '' )
    {
        oResult.strError = strError;
        return( oResult );
    }

    //---------------------------------------------------------------------------------------------------------------------
    // now we build a geared motor object from the MLFB data: the geared motor object has a default configuration that is 
    // modified by overwriting some attributes as given with the MLFB data 
    //---------------------------------------------------------------------------------------------------------------------
    
    if(oBase.oParsingData.strProduct == "GETRIEBE.ADAPTER")
    {
        var oSimulatedGearedMotor = buildGear_ADAPTER( oBase.oParsingData, oShortDescriptions, strBaseMLFB, strRegion );
        if ( !oSimulatedGearedMotor )
            return( oResult );
            
        /*	
        strError = _checkGearedAdapterOptions( oSimulatedGearedMotor, oShortDescriptions, oResult.oParsingData, strRegion );
        if ( strError !== '' )
        {
            oResult.strError = strError;
            return( oResult );
        }
        */
        
        var strBrakeSD = getSDFromGroup( oShortDescriptions, 'BREMSE_TYP' );
        var strBrakeVoltageSD = getSDFromGroup( oShortDescriptions, 'BREMSE_SPANNUNG' );
        var strBrakeFunctionRectifierSD = getSDFromGroup( oShortDescriptions, 'BREMSE_FUNKTIONSGLEICHRICHTER' );
        var strBrakeRectifierSD = getSDFromGroup( oShortDescriptions, 'BREMSE_GLEICHRICHTER' );
        var strBrakeControlSD = getSDFromGroup( oShortDescriptions, 'BREMSE_SCHALTUNG' );
        var strBrakeDesignSD = getSDFromGroup( oShortDescriptions, 'BREMSE_AUSFUEHRUNG' );
        var strBrakeCorrosionProtectionSD = getSDFromGroup( oShortDescriptions, 'BREMSE_KORROSIONSSCHUTZ' );
        var strBrakeWearSD = getSDFromGroup( oShortDescriptions, 'BREMSE_BELAG' );
        var strBrakeManualReleaseSD = getSDFromGroup( oShortDescriptions, 'BREMSE_HANDLUEFTUNG' );
        var strBrakeManualReleaseLocationSD = getSDFromGroup( oShortDescriptions, 'BREMSE_HANDLUEFTUNG_LAGE' );
        var strBrakeMonitorSD = getSDFromGroup( oShortDescriptions, 'BREMSE_MONITOR' );
        var strBrakeCableProtectionSD = getSDFromGroup( oShortDescriptions, 'BREMSE_KABELSCHUTZ' );
        var bHandrad = getSDFromGroup( oShortDescriptions, 'MOTOR_HANDRAD' );
        var bSchutzdach = getSDFromGroup( oShortDescriptions, 'MOTOR_SCHUTZDACH' );
        var bRLSP = getSDFromGroup( oShortDescriptions, 'MOTOR_RUECKLAUFSPERRE' );
        var b2tesWellenende = getSDFromGroup( oShortDescriptions, 'MOTOR_ZWEITESWELLENENDE' );
        var strBelueftung = getSDFromGroup( oShortDescriptions, 'MOTORBELUEFTUNG' );
        
        if(strBrakeSD != "" || strBrakeVoltageSD != "" || strBrakeFunctionRectifierSD != "" || strBrakeRectifierSD != "" || strBrakeControlSD != "" || strBrakeDesignSD != "" || strBrakeCorrosionProtectionSD != "" || strBrakeWearSD != "" || strBrakeManualReleaseSD != "" || strBrakeManualReleaseLocationSD != "" || strBrakeMonitorSD != "" || strBrakeCableProtectionSD != "" || bHandrad || bSchutzdach || bRLSP || b2tesWellenende || strBelueftung != "")
            oResult.strError = translate("ERRORMSG_PARSER_SIMOGEAR_2KJ____WRONG_SD");
        
        
        
    }
    else if(oBase.oParsingData.strProduct == "GETRIEBEMOTOR")
    {
        var oSimulatedGearedMotor = buildGearedMotor( oBase.oParsingData, oShortDescriptions, strBaseMLFB, strRegion );
        if ( !oSimulatedGearedMotor )
            return( oResult );
            
        //---------------------------------------------------------------------------------------------------------------------
        // checking the geared motor is buildable and if the geared motor configuration options match together
        //---------------------------------------------------------------------------------------------------------------------
        /*
        strError = _checkGearedMotor( oSimulatedGearedMotor, oBase.oParsingData );
        if ( strError !== '' )
        {
            oResult.strError = strError;
            return( oResult );
        }
        
        strError = _checkGearedMotorOptions( oSimulatedGearedMotor, oShortDescriptions, oResult.oParsingData, strRegion );
        if ( strError !== '' )
        {
            oResult.strError = strError;
            return( oResult );
        }
        */
    }
    // parsing succeeds - transfer the constructed geared motor object
    oResult.oProduct = oSimulatedGearedMotor;
    
    return( oResult );
}

function MOTOX_parse_CheckforKeepConfiguration( aTokens, strFreetext, bnotfromProjectList, incompleteSizerMLFB, KeepConfiguration )
{
    var oResult = {
        strError: '',
        oParsingData: null,
        oShortDescriptions: null
    };
    
    var strBaseMLFB = aTokens[0];
    if ( isMotoxNMLFB( strBaseMLFB ) )
    {
        // Motox-N cannot be parsed by this function. You must have caught MOTOX-N geared motors before and submitted them to some conversion routine
        oResult.strError = translate( 'ERRORMSG_MLFB_SIMOGEAR_2KJ___PARSE_NO_MOTOXN' );
        return( oResult );
    }
    
    
    //---------------------------------------------------------------------------------------------------------------------
    // freetext analysis
    // since we need some information for necessary additions while checking the base MLFB, we must build the 
    // necessary structures before analysing the base MLFB
    //---------------------------------------------------------------------------------------------------------------------
    var oShortDescriptions = buildShortDescriptions( aTokens, strFreetext, '2KJ8_KURZANGABEN_DB' );	
    if ( oShortDescriptions.strError )
    {
        oResult.strError = oShortDescriptions.strError;
        return( oResult );
    }

    oResult.oShortDescriptions = oShortDescriptions;
    
    //---------------------------------------------------------------------------------------------------------------------
    // now, we do some syntactical checking on the MLFB 
    // if this function is called, we know for sure that the first four digits are '2KJ3'
    //---------------------------------------------------------------------------------------------------------------------
    var oBase = _checkBaseMLFB( strBaseMLFB, oShortDescriptions, "TUB", incompleteSizerMLFB, KeepConfiguration );
    oResult.strError = oBase.strError;
    oResult.oParsingData = oBase.oParsingData;
    
    return( oResult );
}

function parseSetValues(oParse, setAttributes)
{
    var oRetObject = {};
    oRetObject.GearData = {};
    oRetObject.GearData.oRow = {};
    
    oRetObject.BrakeData = {};
    oRetObject.BrakeData.oRow = {};
    
    for ( var Attribute in oParse.oProduct.BrakeData )
    {
        if (typeof oParse.oProduct.BrakeData[Attribute] != "object")
        {
            logtraceln("Attribute:"+Attribute+" set to: "+oParse.oProduct.BrakeData[Attribute]);
            oRetObject.BrakeData.oRow[Attribute] = oParse.oProduct.BrakeData[Attribute];
        }
    }
    
    for ( var Attribute in oParse.oProduct.GearData )
    {
        if (typeof oParse.oProduct.GearData[Attribute] != "object")
        {
            logtraceln("Attribute:"+Attribute+" set to:"+oParse.oProduct.GearData[Attribute]);
            oRetObject.GearData.oRow[Attribute] = oParse.oProduct.GearData[Attribute];
        }
    }
    
    if(setAttributes)
    {
        logtraceln("Set all Attributes from Parser in the configuration");
        for ( var Attribute in oParse.oProduct.oConfig )
        {
            if (typeof oParse.oProduct.oConfig[Attribute] != "object")
            {
                logtraceln("Attribute:"+Attribute+" set to:"+oParse.oProduct.oConfig[Attribute]);
                ECL_SelectValue(Attribute+"_2KJ8", oParse.oProduct.oConfig[Attribute]);
            }
        }
        
        for ( var Attribute in oParse.oProduct )
        {
            if (typeof oParse.oProduct[Attribute] != "object")
            {
                logtraceln("Attribute:"+Attribute+" set to:"+oParse.oProduct[Attribute]);
                ECL_SelectValue(Attribute+"_2KJ8", oParse.oProduct[Attribute]);
            }
        }
    }
    
    return oRetObject;
}

function isKA_M23inMLFB(aArray)
{
    for(var i=0;i<aArray.length;i++)
    {
        if(aArray[i]=="M23")
            return true;
    }
    
    return false;
}

function isKA_N38inMLFB(aArray)
{
    for(var i=0;i<aArray.length;i++)
    {
        if(aArray[i]=="N38")
            return true;
    }
    
    return false;	
}

function ProcessMountingPositionContextChanged(oParsing, strBaseTypeMLFB)
{
    if(find(oParsing.strBaseMLFB.left(16), ".") < 0)
    {
    
        // Überprüfen, ob sich der Context geändert hat
        var oShortDescriptions = buildShortDescriptions( oParsing.aTokens, "", '2KJ8_KURZANGABEN_DB' );	
        var strEinbaulage_KA_old=getSDFromGroup( oShortDescriptions, "GETRIEBE_BAUFORM" );
        
        // "alte" MLFB hat welchen Getriebe-TYP?
        var bFlachoderStirnradGetriebe_old=false;
        var bNICHTFlachStirnradGetriebe_old=false;
        
        var strGearType = oParsing.strBaseMLFB.substring( 4, 5 );
        if(strGearType=="0"||strGearType=="1"||strGearType=="2"||strGearType=="3"||strGearType=="4")
            bFlachoderStirnradGetriebe_old=true;
        else
            bNICHTFlachStirnradGetriebe_old=true;
        
        
        // neue BaseTypeMLFB hat welchen Getriebe-TYP?
        var bFlachoderStirnradGetriebe_BT=false;
        var bNICHTFlachStirnradGetriebe_BT=false;
        
        var strGearType = strBaseTypeMLFB.substring( 4, 5 );
        if(strGearType=="0"||strGearType=="1"||strGearType=="2"||strGearType=="3"||strGearType=="4")
            bFlachoderStirnradGetriebe_BT=true;
        else
            bNICHTFlachStirnradGetriebe_BT=true;
        
        /*
        if(bFlachoderStirnradGetriebe_old==bFlachoderStirnradGetriebe_BT && bNICHTFlachStirnradGetriebe_old==bNICHTFlachStirnradGetriebe_BT)
        {
            // Änderung des Getriebetyps ist nicht relevant, Mounting Position kann weiter verwendet werden
            return oParsing;
        }
        */
        
        //Welcher Mounting Position entspricht der Kontext ang´hand des Getriebetype der neuen BaseType MLFB
        var iMountingPosition = parseInt(ECL_GetValue("ECSI_CX_MountingPosition"));
        var strCX_Bauform="";
        var strCX_Abtriebsseite="";
        var strCX_BAUFORM_EGPOS="";
        
        if(isNaN(iMountingPosition))
        {
            iMountingPosition = 1;
        }
        
        if(iMountingPosition < 7)
        {
            strCX_Bauform = "M"+iMountingPosition;
            
            if(bFlachoderStirnradGetriebe_BT)
            {
                strCX_Abtriebsseite = "-";
                strCX_BAUFORM_EGPOS = strCX_Bauform;
            }
            else
            {
                strCX_Abtriebsseite = "A";
                strCX_BAUFORM_EGPOS = strCX_Bauform + "-A";
            }
        }
        else
        {
            strCX_Bauform = "M"+(iMountingPosition-6);
                
            if(bFlachoderStirnradGetriebe_BT)
            {
                strCX_Abtriebsseite = "-";
                strCX_BAUFORM_EGPOS = strCX_Bauform;
            }
            else
            {
                strCX_Abtriebsseite = "B";
                strCX_BAUFORM_EGPOS = strCX_Bauform + "-B";
            }
        }
        
        //Bestimmung der neuen Mounting Position der KB aus der neuen BaseType MLFB
        var strSelect = "SELECT MLFB_OPTION FROM MLFB8_OPTIONEN WHERE ";
        strSelect=strSelect+"MERKMAL_1='EGPOS' AND WERT_1='"+strCX_BAUFORM_EGPOS+"' ";
        var aResult = Object.fromDatabase( "2KJ8_DB", strSelect );
        
        var strEinbaulageContext_KA="";
        if(aResult.length>0)
            strEinbaulageContext_KA=aResult[0].MLFB_OPTION;
        else
            logtraceln("Fehler bei der Überprüfung des Contextes - Bestimmung KA aus Context; SQL="+strSelect);
        
        if(strEinbaulage_KA_old!="" && strEinbaulageContext_KA != "")
        {
            if(strEinbaulage_KA_old != strEinbaulageContext_KA)
            {
                //Context hat sich geändert
                for(var i=0;i<oParsing.aTokens.length;i++)
                {
                    if(oParsing.aTokens[i] == strEinbaulage_KA_old)
                        oParsing.aTokens[i] = strEinbaulageContext_KA;
                }
                
                //Einbaulage wurde angepasst, MLFB kann sich ändern!
                ECL_SetValue("ECSI_MLFB_CAN_CHANGE_2KJ8", "TRUE");
                return oParsing;
            }
            else
            {
                //Context ist gleich, wir müssen nichts tun
                return oParsing;
            }
        }
        else
        {
            logtraceln("Fehler bei der Überprüfung des Contextes - eine der KA's konnte nicht bestimt werden!");
            return false;
        }
    }
    else
        return oParsing;
}