//@legacy
//#mode(full-ecma)


//@import "Parser.common.js"


/**
 * @category   Parser
 * @package    Core
 * @copyright  Copyright (c) 2009 Plan Software GmbH (http://www.plansoft.de)
 */


/**
 * @constructor
 * build the short description structure from freetext and short description tokens
 *
 * @param aTokens - MLFB token parts from tokenizing step
 * @param strFreetext - free text for short descriptions
 * @param strDatabase - database name where to lookup short description information
 *
 * @return object - object with short description information
 *                  strWarning (optional): additional warning message
 *                  strError (optional): additional error message (only available in case an error occured)
 *                    each attribute name is a short description
 *                    each attribute value is an object with attributes group, freetext (the freetext string may be empty) and text_id
 *                  example: oSD['L1Y'] = { group: 'SEALING', freetext: '20 °C', text_id: IFC_SONSTIGE)
 */
function buildShortDescriptions( aTokens, strFreetext, strDatabase )
{
    var oSD = {};
    var strError = '';
    var strWarning = '';
    
    // build short descriptions structure for freetext
    var oYDescriptions = parseFreetext( strFreetext,true );
    var strText = '';
    for( var p in oYDescriptions )
    {
        if ( p === 'strError' )
            continue;
            
        strText = oYDescriptions[p];
        oYDescriptions[p] = { group: '', freetext: strText };
    }
    if ( oYDescriptions.strError )
        strError = oYDescriptions.strError;
    
    // add short descriptions from tokenizing process
    var oDescriptions = {};
    var strToken = '';
    if(aTokens !== undefined){
        for ( var i = 1; i < aTokens.length; ++i )
        {
            strToken = aTokens[i];
            if ( strToken in oDescriptions )
                strError += ( strError ? ', ' : '' ) + translate( 'PARSING_ERROR_2KJ___DOUBLE_DESCRIPTION' ) + ' ' + strToken;
            

            // Nur existierende KA's übernehmen
            var strStartConfiguration=ECL_GetValue("ECSI_IS_START_CONFIGURATION");
            if(strStartConfiguration=="1")
            {
                var strSelect_Test;
                strSelect_Test = "SELECT OPTION_ID, DICT_ID FROM OPTION_VALUES WHERE KURZANGABE = '" + strToken + "' ";
                if ( dbselect( strDatabase, strSelect_Test, 'RS_OPTION_KA_TEST' ) > 0 )
                {
                    oDescriptions[strToken] = { group: '', freetext: '' };
                }
            }else
            {
                oDescriptions[strToken] = { group: '', freetext: '' };
            }
        }
    }
    
    // possible source of errors: a short description that has a freetext is not mentioned in the tokens
    // this one [a short description that is mentioned in the tokens has no freetext although is must have one] is tested in another step
    for( var p in oYDescriptions )
    {
        if ( p === 'strError' )
            continue;
            
        if ( !( p in oDescriptions ) )
            strError += ( strError ? ', ' : '' ) + translate( 'ERRORMSG_2KJ___MLFB_KA_NOT_DEFINED' ) + ' ' + p;
        oDescriptions[p] = oYDescriptions[p];
    }

    // now find the group for each short description
    if ( strError === '' )
    {
        var strText, strSelect, strGroup;
        var oGroups = {};
        for( p in oDescriptions )
        {
            strSelect = "SELECT OPTION_ID, DICT_ID FROM OPTION_VALUES WHERE KURZANGABE = '" + p + "' ";
            if ( dbselect( strDatabase, strSelect, 'RS_OPTION' ) > 0 )
            {
                strGroup = dbgetstring( 'RS_OPTION', 'OPTION_ID' );
                if ( strGroup in oGroups )
                    strError += ( strError ? ', ' : '' ) + translate( 'PARSING_ERROR_2KJ___DOUBLE_DESCRIPTION' ) + ' ' + p;
                else oGroups[strGroup] = p;
                oDescriptions[p].group = strGroup;
                oDescriptions[p].text_id = dbgetstring( 'RS_OPTION', 'DICT_ID' );
            }
            else
            {
                strText = replace( translate( 'PARSING_ERROR_2KJ___NECESSARY_DESCRIPTION' ), '%s', p );
                strError += ( strError ? ', ' : '' ) + strText;
            }
        }
    }
    oDescriptions.strError = strError;
    oDescriptions.strWarning = strWarning;
    
    return( oDescriptions );
}


/**
 * @constructor
 * get a short description from a short description group
 *
 * @param oShortDescriptions - object with short description information
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group, freetext (the freetext string may be empty) and text_id
 *                             example: oSD['L1Y'] = { group: 'SEALING', freetext: '20 °C' }
 * @param strGroup - name of the group
 *
 * @return string - the short description (may be empty if none for this group exists)
 */
function getSDFromGroup( oShortDescriptions, strGroup )
{
    var strSD = '';
    
    for( var p in oShortDescriptions )
    {
        if(oShortDescriptions[p])
        {
            if ( oShortDescriptions[p].group === strGroup )
            {
                strSD = p;
                break;
            }
        }
    }
    
    return( strSD );
}


/**
 * @constructor
 * test if any short description is present in a short description group
 *
 * @param oShortDescriptions - object with short description information
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group, freetext (the freetext string may be empty) and text_id
 *                             example: oSD['L1Y'] = { group: 'SEALING', freetext: '20 °C' }
 * @param strGroup - name of the group
 *
 * @return boolean - false: no short description is present in this group
 *                   true: one short description is present in this group
 */
function isSDPresent( oShortDescriptions, strGroup )
{
    return( Parser.getSDFromGroup( oShortDescriptions, strGroup ) !== '' );
}


/**
 * @constructor
 * build text for short description and corresponding option
 *
 * @param oShortDescriptions - object with short description information
 *                             each attribute name is a short description
 *                             each attribute value is an object with attributes group, freetext (the freetext string may be empty) and text_id
 *                             example: oSD['L1Y'] = { group: 'SEALING', freetext: '20 °C' }
 * @param strShortDescription - id of short description
 * @param strDatabase - database name where to lookup short description information
 *
 * @return string - description line for short description
 */
function getSDDescriptionLine( oShortDescriptions, strShortDescription, strDatabase )
{
    if ( ( strShortDescription === '' ) || ( strShortDescription === 'Y99' ) || !( strShortDescription in oShortDescriptions ) )
        return( '' );
        
    // build text for short description
    // if the content of the column OPTION_VALUES.DICT_ID is surrounded by {}, the description text is only this content, not translated, without the braces
    // if the content of the column OPTION_VALUES.DICT_ID is surrounded by [], the description text is the translated dict_id of the short description group and this content, not translated, without the braces
    // if the content of the column OPTION_VALUES.DICT_ID is not surrounded by any braces, the description text is the translated dict_id of the short description group and the translated content
    var strTextID = oShortDescriptions[strShortDescription].text_id;
    if ( strTextID === '' )
        strTextID = oShortDescriptions[strShortDescription].group + '_' + strShortDescription;
    var strSDText;
    if ( ( strTextID !== '' ) && ( strTextID.charAt( 0 ) === '{' ) )
        strDescriptionText = strTextID.substring( 1, strTextID.length - 1 );
    else
    {
        if ( strTextID.charAt( 0 ) === '[' )
            strSDText = strTextID.substring( 1, strTextID.length - 1 );
        else strSDText = translate( strTextID );
    
        // build text for option
        strTextID = oShortDescriptions[strShortDescription].group;
        var strDBTable = 'RS_OPTIONS_TEXT';
        var strSelect = "SELECT DICT_ID FROM OPTIONS WHERE OPTION_ID ='" + strTextID + "' and AKTIV = '1' ";
        if ( dbselect( strDatabase, strSelect, strDBTable ) > 0 )
            strTextID = dbgetstring( strDBTable, 'DICT_ID' );
        var strOptionText = translate( strTextID );

        // build description text
        strOptionText = trimright( strOptionText, ' ' );
        strOptionText = trimright( strOptionText, ':' );
        var strDescriptionText = strOptionText;
        if ( strSDText != '' )
            strDescriptionText += ' : ' + strSDText;
    }
    
    return( strDescriptionText );
}


/**
 * @constructor
 * get the value of the freetext, performs unit conversion if necessary
 *
 * @param strFreetext - encoded freetext (form {xxx: value unit})
 * @param strUnit (optional) - unit of a techvalue
 *
 * @return string - freetext value
 */
function getFreetextValue( strFreetext, strUnit )
{
    if ( !strFreetext )	// undefined freetext -> empty value
        return( '' );
        
    var ft = new StringHandler( strFreetext );
    var aFreetext = ft.split( ' ' );
    if ( !strUnit || ( aFreetext.length === 0 ) )
        return( strFreetext );
        
    //	aFreetext[0] --> value
    //	aFreetext[1] --> unit or empty

    // unit conversion not possible or necessary
    if ( ( aFreetext.length === 1 ) || ( utf8toansi(aFreetext[1]) === strUnit ) ) {
        var x = (aFreetext[0] - 0);
        // -mw- 10.8.2010 ... wir müssen den Fall abfangen z.B. {Y01: 0C}, also inkorrekte Syntax
        // besser wäre natürlich eine Exception, aber die müssten wir dann an über 70 Stellen abfangen !?
        // Siehe auch Parser.FZG.parse.es, zeile 229 ff
        if (isNaN(x))
            return( '-9999' );		
        return( aFreetext[0] );
    }

    // if units differ only by case, we consider them identical (no need to do any conversion)
    strUnit = strUnit.toLowerCase();
    var strFreetextUnit = aFreetext[1].toLowerCase();
    if ( strUnit === strFreetextUnit )
        return( aFreetext[0] );

    var strValue = aFreetext[0];
    var dValue = strValue - 0;
        
    // try to perform unit conversion for power
    if ( ( strUnit === 'kw' ) && ( ( strFreetextUnit === 'kw' ) || ( strFreetextUnit === 'hp' ) ) )
    {
        if ( strFreetextUnit === 'hp' )
        {
            dValue = dValue * 0.7457;
            strValue = formatstring( dValue, '%.2f' );
        }
        return( strValue );
    }
    
    // try to perform unit conversion for torque
    if ( ( strUnit === 'nm' ) && ( ( strFreetextUnit === 'nm' ) || ( strFreetextUnit === 'lb.in' ) || ( strFreetextUnit === 'lb.ft' ) ) )
    {
        if ( strFreetextUnit === 'lb.in' )
        {
            dValue = dValue * 0.11298483;
            strValue = formatstring( dValue, '%.2f' );
        }
        else if ( strFreetextUnit === 'lb.ft' )
        {
            dValue = dValue * 1.355818;
            strValue = formatstring( dValue, '%.2f' );
        }
        return( strValue );
    }
    
    // try to perform unit conversion for temperature
    if ( ( strUnit === 'c' ) &&
         ( ( strFreetextUnit === ansi2utf8( '°' ) ) || ( strFreetextUnit === ansi2utf8( '°c' ) ) || ( strFreetextUnit === ansi2utf8( '°f' ) ) || 
           ( strFreetextUnit === '°' ) || ( strFreetextUnit === '°c' ) || ( strFreetextUnit === '°f' ) || 
           ( strFreetextUnit === 'c' ) || ( strFreetextUnit === 'f' ) ) )
    {
        if ( ( strFreetextUnit === 'f' ) || ( strFreetextUnit === ansi2utf8( '°f' ) ) || ( strFreetextUnit === '°f' ) )
        {
            dValue = (5.0/9.0) * ( dValue - 32.0 );
            strValue = formatstring( dValue, '%.2f' );
        }
        return( strValue );
    }
    
    // try to perform unit conversion for angle degree
    if ( ( strUnit === '°' ) && ( strFreetextUnit === ansi2utf8( '°' ) ) )
        return( strValue );
    
    // other units are not supported at the moment
    return( '' );
}


/**
 * @constructor
 * converts a short description to gear data value and sets it to the gear or gear configuration
 *
 * @param strShortDescription - short description to convert and set
 * @param oGear - gear object to adjust
 * @param strDatabase - database name where to lookup short description information
 */
function setShortDescriptionToGear( strShortDescription, oGear, strDatabase )
{
    // assemble select statement
    var strSelect = "SELECT PROPERTY, PROPERTY_VALUE FROM OPTION_VALUES where KURZANGABE='" + strShortDescription + "' ";
    
    // get data from database
    var strProperty, strPropertyValue;
    var strDBResult = 'SHORT_DESCRIPTIONS';
    if ( dbselect( strDatabase, strSelect, strDBResult ) > 0 )
    {
        while ( dbiseof( strDBResult ) == 0 )
        {
            // get value location of short description position in the object
            strProperty = dbgetstring( strDBResult, 'PROPERTY' );
            strPropertyValue = dbgetstring( strDBResult, 'PROPERTY_VALUE' );
            if ( strProperty !== '' )
            {
                oGear.oConfig[strProperty] = strPropertyValue;
            }
            
            dbskip( strDBResult );
        }
    }
}

function getNotActiveShrtDescriptions(strOptionId,strDatabase){
    var aNot = new Array();
    var strSelect = "SELECT KURZANGABE FROM OPTION_VALUES where AKTIV = '0' AND OPTION_ID like '"+strOptionId+"%' ";
    
    var strDBResult = 'SHORT_DESCRIPTIONS';
    if ( dbselect( strDatabase, strSelect, strDBResult ) > 0 )
    {
        while ( dbiseof( strDBResult ) == 0 )
        {
            // get value location of short description position in the object
            aNot.push( dbgetstring( strDBResult, 'KURZANGABE' ));			
            
            dbskip( strDBResult );
        }
    }
    
    return aNot;
}


