//@legacy
//#mode(full-ecma)


/**
 * @category   Parser
 * @package    Core
 * @copyright  Copyright (c) 2009-2013 Plan Software GmbH (http://www.plansoft.de)
 */

/**
 * checks if the given mlfb has the structure of a customer-specific gear
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false: not a valid customer-specific gear
 *                   true: valid customer-specific gear
 */
function isCustomerSpecificGear( strMLFB )
{
    if ( strMLFB.length < 11 )
        return( false );

    var strPrefix = strMLFB.substring( 0, 4 );
    var nATN = strMLFB.substring( 4, 6 ) - 0;
    var strCustID = strMLFB.substring( 8, 10 );
    return( ( strPrefix === '2LP0' ) && ( strCustID === 'YY' ) && ( nATN >= 2 ) && ( nATN <= 11 ) );
}


/**
 * checks if the MLFB describes a coupling product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a coupling product
 *                   true - MLFB describes a coupling product
 */
function isCouplingMLFB( strMLFB )
{
    if ( strMLFB.substring( 0, 3 ) === '2LC' ) {
        var c = strMLFB.charAt( 3 );
        if ( c.isDigit() )
            return( true );
    }
    
    return( false );
}


/**
 * checks if the MLFB describes a FZG product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a FZG product
 *                   true - MLFB describes a FZG product
 */
function isFZGMLFB( strMLFB )
{	
    if (( strMLFB.substring( 0, 3 ) === '2LP' ) && ( ! Parser.isFSIGMLFB (strMLFB) ))
    {
        var c = strMLFB.charAt( 3 );
        if ( ( c !== '9' ) && ( c !== '8' ) )
            return( !Parser.isSIPMLFB( strMLFB ) );
    }
    
    return( false );
}

/**
 * checks if the MLFB describes a FSIG gear product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a FSIG product
 *                   true - MLFB describes a FSIG product
 */
function isFSIGMLFB( strMLFB )
{	
    return( strMLFB.substring( 0, 4 ) === '2LP2' );
}


/**
 * checks if the MLFB describes a SIP gear product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a SIP product
 *                   true - MLFB describes a SIP product
 */
function isSIPCustomerMLFB( strMLFB )
{	
    return((strMLFB.substring( 0, 6 ) === '2LP969') && (strMLFB.substring( 8, 9 ) === 'Y') && (strMLFB.substring( 9, 10 ) === 'Y'));
}

/**
 * checks if the MLFB describes a SIP gear product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a SIP product
 *                   true - MLFB describes a SIP product
 */
function isSIPMLFB( strMLFB )
{	
    return(strMLFB.substring( 0, 6 ) === '2LP069');
}


/**
 * checks if the MLFB describes a trunk product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a trunk product
 *                   true - MLFB describes a trunk product
 */
function isTrunkMLFB( strMLFB )
{
    var strMLFBStart = strMLFB.substring( 0, 3 );
    if ( strMLFBStart === '2LP' )
    {
        var cGeneration = strMLFB.charAt( 3 );
        return( ( cGeneration === '8' ) || ( cGeneration === '9' ) );
    }
    else if ( strMLFBStart === '2LX' )
        return( true );
    
    return( false );
}


/**
 * checks if the MLFB describes a FFA single part
 * update 20141215 new FFA Numbers with "A5E"
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a FFA single part
 *                   true - MLFB describes a FFA single part
 */
function isFFAOrderNumber( strMLFB )
{
    var strOrderNumber = trimright( trimleft( toupper( strMLFB ), ' ' ), ' ' );
    if ( strOrderNumber.substring( 0, 3 ) === 'FFA' )
    {
        if ( strOrderNumber.substring( 0, 4 ) === 'FFA:' )
            strOrderNumber = replace( strOrderNumber, 'FFA:', '' );
        else strOrderNumber = replace( strOrderNumber, 'FFA', '' );
    }
    
    var bIsOnlyNumber = isOnlyNumbers( strOrderNumber ) && strOrderNumber.length <= 12;
    var bIsA5E		  = (strOrderNumber.substring(0,3)=='A5E') && strOrderNumber.length == 11;

    return( bIsOnlyNumber || bIsA5E );
}


/**
 * checks if the MLFB describes a DCC order number
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a DCC order number
 *                   true - MLFB describes a DCC order number
 */
function isDCCOrderNumber( strMLFB )
{
    var strOrderNumber = trimright( trimleft( toupper( strMLFB ), ' ' ), ' ' );
    return( strOrderNumber.substring( 0, 3 ) === 'HGL' );
}


/**
 * tests if a string contains only numbers
 *
 * @param strText - string to test
 *
 * @return boolean - false - string also contains other characters
 *                   true - string contains only numbers
 */
function isOnlyNumbers( strText )
{
    // let's define: empty text doesn't contain numbers
    if ( !strText  )
        return( false );
    
    return( strText.search( /[^0-9]/ ) === -1 );
}


/**
 * checks if the MLFB describes a MOTOX geared motor product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a MOTOX geared motor product
 *                   true - MLFB describes a MOTOX geared motor product
 */
function isMotoxMLFB( strMLFB )
{
    try{
        return( strMLFB.substring( 0, 3 ) === '2KJ' );
    }
    catch(err){
        logtraceln(err.message);
    }
}


/**
 * checks if the MLFB describes a MOTOX-N product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a MOTOX-N product
 *                   true - MLFB describes a MOTOX-N product
 */
function isMotoxNMLFB( strMLFB )
{
    return( strMLFB.substring( 0, 4 ) === '2KJ1' );
}


/**
 * checks if the MLFB describes a SimoGear product
 *
 * @param strMLFB - MLFB to check
 *
 * @return boolean - false - MLFB doesn't describe a SimoGear product
 *                   true - MLFB describes a SimoGear product
 */
function isSimoGearMLFB( strMLFB )
{
    if(strMLFB.substring( 0, 4 ) === '2KJ8')
        return( true);
    else
        return( false);
}


