//@legacy
//#mode(full-ecma)

/**
 * @category   MLFB generation
 * @package    product logic
 * @copyright  Copyright (c) 2009-2013 Plan Software GmbH (http://www.plansoft.de)
 */

/**
 * @constructor
 * inserts '-' signs at the appropriate places in an MLFB, if not present (works only for 16digit MLFBs)
 *
 * @param strBaseMLFB - MLFB that is to beautifiy
 *
 * @return string - beautified MLFB
 */
function beautifyMLFB16( strBaseMLFB )
{
    var strTestMLFB = replace( strBaseMLFB, '-', '' );
    
    // now, MLFB doesn't contain beautifier
    strTestMLFB = strTestMLFB.substring( 0, 7 ) + '-' + strTestMLFB.substring( 7, 12 ) + '-' + strTestMLFB.substring( 12 );
    
    return( strTestMLFB );
}


/**
 * @constructor
 * convert array of fully specified short descriptions to various short description texts
 *
 * @param aShortDescriptions - fully specified short descriptions
 * @param strYTextPrefix (optional) - prefix for translated text part of Y-descriptions
 *
 * @return object - formatted short description texts
 *                  strStrippedShortDescriptions - all short descriptions where all clear texts for Yxx- descriptions have been stripped
 *                  strFormattedShortDescriptions - clear texts for Yxx- descriptions that have been formatted to {<SHORT_DESCRIPTION>: <TEXT>}
 *                  strYDescription - clear texts for Y descriptions, one per row, rows are separated with #
 *
 * @remarks the function may change the parameter array aShortDescriptions; do not use it after the function returns
 */
function buildFormattedShortDescriptions( aShortDescriptions, strYTextPrefix )
{
    var oSD = {
        strStrippedShortDescriptions : '',
        strFormattedShortDescriptions : '',
        strYDescription : ''
    };
    
    // filter out short descriptions with additional texts
    var strSD = '', pos, strShortSD, strYText;
    var aFormatted = [], aYDescription = [];
    for( var i = 0; i < aShortDescriptions.length; ++i )
    {
        strSD = aShortDescriptions[i];
        pos = strSD.indexOf( "'" );
        // strip and reformat the additional texts
        if ( pos > -1 )
        {
            strShortSD = strSD.substring( 0, pos );
            strYText = strSD.substring( pos + 1, strSD.length - 1 );
            
            aShortDescriptions[i] = strShortSD;
            if ( strShortSD.charAt( 0 ) === '+' )
                strShortSD = strShortSD.substring( 1 );
            aFormatted.push( '{' + strShortSD + ':' + strYText + '}' );
            aYDescription.push( translate( strYTextPrefix + strShortSD ) + '[' + strShortSD + ']:\t' + strYText );
        }
    }
    
    // convert the resulting arrays to strings
    strSD = aShortDescriptions.join( ' ' );
    if ( strSD.substring( 0, 1 ) == '+' )
        strSD = strSD.substring( 1 );
    oSD.strStrippedShortDescriptions = strSD;
    if ( aFormatted.length > 0 )
        oSD.strFormattedShortDescriptions = aFormatted.join( '' );
    if ( aYDescription.length > 0 )
    {
        oSD.strYDescription = aYDescription.join( '#' );
        oSD.strYDescription = replace( replace( oSD.strYDescription, '<sub>', '' ), '</sub>', '' );
    }
    
    return( oSD );
}


/**
 * @constructor
 * get the short descriptions for various gear options, which can be computed automatically by a simple lookup in a database
 * 
 * @param oGear - gear object
 *
 * @return array - each index contains a string with one short description (may be empty)
 */
function getShortDescriptionsFromDatabase( oGear, oExceptions, strDatabase )
{
    // use the database table OPTION_VALUES, cycle through all available short 
    // descriptions and look if the appropriate property in the gear configuration object is set	
    var aShortDescriptions = [];
    var strSelect = "SELECT * FROM OPTION_VALUES ";
    var strResultTable = 'SHORT_DESCRIPTIONS';
    var strOptionID, strDBProperty, strDBValue, strCurrentValue, strSD;
    if ( dbselect( strDatabase, strSelect, strResultTable ) > 0 )
    {
        while ( !dbiseof( strResultTable ) )
        {
            strOptionID = dbgetstring( strResultTable, 'OPTION_ID' );
            logtraceln(strOptionID);
            // only proceed if the short description for this group not handled
            if ( !( strOptionID in oExceptions ) )
            {
                // get value location of short description position in the store
                strDBProperty = dbgetstring( strResultTable, 'PROPERTY' );
                strDBValue = dbgetstring( strResultTable, 'PROPERTY_VALUE' );
                if ( ( strDBProperty !== '' ) && ( strDBValue !== '' ) )
                {
                    if ( ( strOptionID === 'AUFSTELLORT' ) || ( strOptionID === 'AUFSTELLHOEHE' ) )
                        strCurrentValue = oGear[strDBProperty];
                    else strCurrentValue = String( oGear.oConfig[strDBProperty] );
                    if ( strCurrentValue === strDBValue )
                    {
                        // get short description
                        strSD = dbgetstring( strResultTable, 'KURZANGABE' );
                        aShortDescriptions.push( '+' + strSD );
                        // mark the group of that short description as handled
                        oExceptions[strOptionID] = strSD;
                    }
                }
            }
            dbskip( strResultTable );
        }
    }
    
    return( aShortDescriptions );
}


/**
 * @constructor
 * get the short descriptions for various gear options, which can be computed automatically by a simple lookup in a database
 * 
 * @param oComponent - drivetrain component object
 *
 * @return array - each index contains a string with one short description (may be empty)
 */
function getShortDescriptionsDrivetrainComponentFromDatabase( oGear, oExceptions, strDatabase )
{
    // use the database table OPTION_VALUES, cycle through all available short 
    // descriptions and look if the appropriate property in the gear configuration object is set	
    var aShortDescriptions = [];
    var strSelect 		= "SELECT * FROM OPTION_VALUES ";
    var strResultTable 	= 'SHORT_DESCRIPTIONS';
    var strOptionID, strDBProperty, strDBValue, strCurrentValue, strSD;
    if ( dbselect( strDatabase, strSelect, strResultTable ) > 0 )
    {
        while ( !dbiseof( strResultTable ) )
        {
            strOptionID = dbgetstring( strResultTable, 'OPTION_ID' );
            // only proceed if the short description for this group not handled
            if ( !( strOptionID in oExceptions ) )
            {
                // get value location of short description position in the store
                strDBProperty = dbgetstring( strResultTable, 'PROPERTY' );
                strDBValue = dbgetstring( strResultTable, 'PROPERTY_VALUE' );
                if ( ( strDBProperty !== '' ) && ( strDBValue !== '' ) )
                {
                    if ( ( strOptionID === 'AUFSTELLORT' ) || ( strOptionID === 'AUFSTELLHOEHE' ) )
                        strCurrentValue = oGear[strDBProperty];
                    else strCurrentValue = String( oGear[strDBProperty] );
                    if ( strCurrentValue === strDBValue )
                    {
                        // get short description
                        strSD = dbgetstring( strResultTable, 'KURZANGABE' );
                        aShortDescriptions.push( '+' + strSD );
                        // mark the group of that short description as handled
                        oExceptions[strOptionID] = strSD;
                    }
                }
            }
            dbskip( strResultTable );
        }
    }	
    
    return( aShortDescriptions );
}

/**
 * build MLFB as string (base MLFB + '-Z' + short descriptions without clear text)
 *
 * @param oMLFB - object
 *                strBaseMLFB - base MLFB
 *                strStrippedShortDescriptions - all short descriptions where all clear texts for Yxx- descriptions have been stripped
 *                strFormattedShortDescriptions - clear texts for Yxx- descriptions that have been formatted to {<SHORT_DESCRIPTION>: <TEXT>}
 *                strYDescription - clear texts for Y descriptions, one per row, rows are separated with #
 *
 * @return string - complete MLFB
 */
function MLFBAsString( oMLFB )
{
    if ( !oMLFB || !oMLFB.strBaseMLFB )
        return( '' );

    var strMLFB = oMLFB.strBaseMLFB;
    if ( oMLFB.strStrippedShortDescriptions )
        strMLFB += '-Z ' + oMLFB.strStrippedShortDescriptions;
        
    return( strMLFB );
}

