//@legacy
//#mode(full-ecma)


/**
 * @category   Parser
 * @package    Core
 * @copyright  Copyright (c) 2009 Plan Software GmbH (http://www.plansoft.de)
 */


/**
 * tokenize the MLFB with a given separator; the short descriptions may be separated with spaces or may arise in several cohesive blocks. the base MLFB must be 
 * separated with a space from the short description blocks
 *
 * @param strMLFB - the whole MLFB that should be tokenized
 * @param bStripFreetext (optional) - false: freetext (if any) should remain part of the tokens
 *                                    true: freetext should be cleared from the tokens
 *
 * @return array - array with MLFB tokens 
 *                 the base MLFB is always in the first index of the array, the short descriptions (if any) follow
 */
function tokenize( strMLFB, bStripFreeText )
{
	var nMLFBLen = strMLFB.length;
	var bMasked = false, bStripText = false, bBaseMLFBFinished = false;
	var strToken = '';
	var cToken = '';
	var aTokens = [];
	
	var i = 0;
	while ( i <= nMLFBLen )
	{
		cToken = strMLFB.charAt( i );
		if ( ( cToken === ' ' ) && !bMasked )					// not a space in quotes
		{
			if ( strToken === ' ' )								// clear multiple spaces
				strToken = '';
			else if ( strToken !== '' )							// a single space is a separator between tokens
			{
				aTokens.push( strToken );
                strToken = '';
			}
			// now we are handling blocks of mlfbs - but the MLFB must always be sparated by a space from the short descriptions
			if ( !bBaseMLFBFinished )
				bBaseMLFBFinished = true;
		}
		else if ( _isQuoteChar( cToken ) && !bMasked )	// text (in double or single quotes) started
		{
			// if we don't want the freetext to be included, we must not add a starting double quote
			// and we must prevent following chars to be added to the token until a closing double quote
			bMasked = true;
			if ( !bStripFreeText )
				strToken += cToken;
			else bStripText = true;
		}
		else if ( _isQuoteChar( cToken ) && bMasked )		// text (in double or single quotes) ended
		{
			// } of quote means } of token
			if ( !bStripFreeText )
				strToken += cToken;
			else bStripText = false;
			// if we don't want the freetext to be included, we must not add a starting double quote
			// and we must prevent following chars to be added to the token until a closing double quote
			if ( strToken !== '' )
			{
				aTokens.push( strToken );
                strToken = '';
	        }
			bMasked = false;
		}
		else
		{
			if ( !bStripText )
			{
				if ( !_isSpecialChar( cToken, bMasked ) || i < 16)
				{
					// the lexer now must accept blocks of short descriptions and must break them into pieces of 3 chars
					strToken += cToken;
					if ( bBaseMLFBFinished && !bMasked && ( strToken.length == 3 ) )	// base mlfb is lexed and string is not masked
					{						
						if ( ( i + 1 ) <= nMLFBLen )
							cToken = strMLFB.charAt( i + 1 );
						if ( !_isQuoteChar( cToken ) )		// don't split a MLFB and quoted freetext
						{
							aTokens.push( strToken );
							strToken = '';
		                }
					}					
				}
			}
		}	
		++i;
	} 
	
	if ( strToken !== '' )
	{
		// push the remaining string as token
		aTokens.push( strToken );
	}
		
	return( aTokens );
}


/**
 * determines if a character is a special char, i.e. a character that should not be included in a token
 *
 * @param cChar - char to examin
 * @param bInQuotes - are we just in double quotes?
 *
 * @return boolean - false - not a special character
 *                   true - a special character
 */
function _isSpecialChar( cChar, bInQuotes )
{
	var bIsSpecial = false;
	if ( !bInQuotes )
	{
		if ( ( cChar === '-' ) ||
			 ( cChar === ',' ) ||
			 ( cChar === 'Z' ) ||
			 ( cChar === ';' ) ||
			 ( cChar === ':' ) ||
			 ( cChar === '_' ) ||
			 ( cChar === '//' ) ||
			 ( cChar === '?' ) ||
			 ( cChar === '*' ) ||
			 ( cChar === '+' ) ||
			 ( cChar === "'" ) ||
			 ( cChar === '/' ) ||
			 ( cChar === '(' ) ||
			 ( cChar === ')' ) ||
			 ( cChar === '[' ) ||
			 ( cChar === ']' ) ||
			 ( cChar === '\t' ) )
			bIsSpecial = true;
	}
	// else characters that are in quotes are never special
	
	return( bIsSpecial );
}


/**
 * determines if a character is a quotation character in a token
 *
 * @param cChar - char to examin
 *
 * @return boolean - false - not a quotation character
 *                   true - a quotation character
 */
function _isQuoteChar( cChar )
{
	return( ( cChar === '\"' ) || ( cChar === "'" ) );
}


