//@legacy
//#mode(full-ecma)


// Retrieve all selected values of passed attribute and iterate over them in order to check, if one of them matches
// passed condition considering operator and expected value.
// Boolean value true is returned if selected attribute value matching condition is found, otherwise false
function compareRuleSelectedValues (ruleCondition)
{
    var allSelectedValues = ECL_GetAllSelectedAttributeValues(ruleCondition.condAttr);
    
    for (var i = 0; i < allSelectedValues.length; i++)
    {
        if (allSelectedValues[i] != undefined)
        {
            var adaptedValue = allSelectedValues[i];
            if (ruleCondition.valueDataType == 'integer')
            {
                adaptedValue = parseInt(adaptedValue);
            }
            else if (ruleCondition.valueDataType == 'double')
            {
                adaptedValue = parseFloat(adaptedValue);
            }
            
            if ((ruleCondition.condOp == '>' && adaptedValue > ruleCondition.condValue) ||
                (ruleCondition.condOp == '>=' && adaptedValue >= ruleCondition.condValue) ||
                (ruleCondition.condOp == '<=' && adaptedValue <= ruleCondition.condValue) ||
                (ruleCondition.condOp == '<' && adaptedValue < ruleCondition.condValue))
            {
                return true;
            }
        }
    }
    
    return false;
}


// Performs compliance check of condition of rule, which references substructure(s) via long distance path. This path
// gets rebuilt recursively, so compliance of each substructure being referenced by condition, is checked.
// Result of check gets returned as boolean value and is also saved to global variable 'substructConditionResults',
// to keep it available for further processing by other routines (see below)
//
//	@param ruleCondition: Object providing the following data of condition of rule as parameters for processing
//				- condAttr: String containing long distance path of condition (i.e. substructure(s) get referenced)
//				- condOp: String of operator of condition used to perform compliance check
//				- valueDataType: Keyword signalizing data type of value of condition, so it gets casted respectively
//						before compliance check is performed
//				- condValue: Value of condition used to perform compliance check. It's not present, if only general
//						evaluation state of attribute referenced by long distance path should be checked
//				- emptyValue: Boolean flag-parameter signalizing, that condition doesn't refer to specific value, so
//						compliance check means only validation of general evaluation state
function checkSubstructCondition (ruleCondition)
{
    if (substructConditionResults[ruleCondition.condAttr] != undefined &&
        substructConditionResults[ruleCondition.condAttr] == true)
    {
        return true;
    }
    
    var substructPathSegments = ruleCondition.condAttr.split('.');
    var substructInstanceIdxs = [];
    var conditionMetAtLeastOnce = false;
    
    var substructuresLoop = function (params) {
        var currentPath = (params.currentPath != '') ? params.currentPath + '.' : params.currentPath;
        currentPath += substructPathSegments[params.pathSegmentIdx];
        
        for (var i = 0; i < ECL_GetCard(currentPath); i++)
        {
            substructInstanceIdxs[params.pathSegmentIdx] = i;
            
            if (params.pathSegmentIdx < substructPathSegments.length - 2)
            {
                substructuresLoop ({
                    pathSegmentIdx: params.pathSegmentIdx + 1,
                    currentPath: currentPath + '.RANGE.' + i
                });
                
                if (conditionMetAtLeastOnce)
                {
                    substructConditionResults[currentPath] = true;
                }
            }
            else
            {
                var conditionMetCurSubstruct = false;
                var exactFullPath = currentPath + '.RANGE.' + substructInstanceIdxs[params.pathSegmentIdx] + '.' + substructPathSegments[params.pathSegmentIdx+1];
                var generalFullPath = currentPath + '.' + substructPathSegments[i+1];
                
                if (!ruleCondition.emptyValue)
                {
                    if (ruleCondition.condOp == '=')
                    {
                        conditionMetCurSubstruct = ECL_ValueSelected(exactFullPath, ruleCondition.condValue);
                    }
                    else if (ruleCondition.condOp == '!=')
                    {
                        conditionMetCurSubstruct = !ECL_ValueSelected(exactFullPath, ruleCondition.condValue);
                    }
                    else if (ruleCondition.condOp == 'like')
                    {
                        conditionMetCurSubstruct = ECL_SelectedAttributeValuesContain(exactFullPath, ruleCondition.condValue);
                    }
                    else if (ruleCondition.condOp == 'not like')
                    {
                        conditionMetCurSubstruct = !ECL_SelectedAttributeValuesContain(exactFullPath, ruleCondition.condValue);
                    }
                    else
                    {
                        conditionMetCurSubstruct = compareRuleSelectedValues({
                            condAttr: exactFullPath,
                            condValue: ruleCondition.condValue,
                            valueDataType: ruleCondition.valueDataType,
                            condOp: ruleCondition.condOp
                        });
                    }
                }
                else
                {
                    if (ruleCondition.condOp == '=')
                    {
                        conditionMetCurSubstruct = ECL_AtLeastOneSelected(exactFullPath);
                    }
                    else
                    {
                        conditionMetCurSubstruct = !ECL_AtLeastOneSelected(exactFullPath);
                    }
                }
                
                // Save result of condition check of current substructure
                substructConditionResults[exactFullPath] = conditionMetCurSubstruct;
                
                if (!conditionMetAtLeastOnce && conditionMetCurSubstruct)
                {
                    conditionMetAtLeastOnce = true;
                }
                
                if (conditionMetAtLeastOnce &&
                    (substructConditionResults[generalFullPath] == undefined ||
                     substructConditionResults[generalFullPath] != true))
                {
                    substructConditionResults[generalFullPath] = true;
                    substructConditionResults[generalFullPath.substring(0, generalFullPath.lastIndexOf('.'))] = true;
                }
            }
        }
    };
    
    substructuresLoop ({
        pathSegmentIdx: 0,
        currentPath: ''
    });
    
    return conditionMetAtLeastOnce;
}


// Handle performance of action of rule, from whose conditions at least one references substructure(s) via long
// distance path. It recursively looks-up result of compliance check (performed by routine 'checkSubstructCondition')
// for each substructure being referenced by these conditions. Then, action gets executed as often as condition is met
// respectively not met according to value of flag-parameter 'ruleMet'
//
//	@param actionParams: Object providing the following properties as parameters for processing
//				- substructCondPaths: Array of Strings representing long distance paths of all respective conditions
//						assigned to rule (i.e. without those conditions accessing only same configuration structure)
//				- action: EasyScript code, which stands for action of rule and should be executed as often as all
//						conditions are met respectively not met
//				- ruleMet: Boolean flag-parameter controlling execution of action if all conditions are met or not met
function performSubstructCondRuleAction (actionParams)
{
    var substructInstanceIdxs = [];
    var conditionsMet = [];
    var conditionsNotMet = [];
    var numExecutions = 1;
    var curConditionIdx;
    
    var substructuresLoop = function (params) {
        var currentPath = (params.currentPath != '') ? params.currentPath + '.' : params.currentPath;
        currentPath += params.substructPathSegments[params.pathSegmentIdx];
        
        for (var i = 0; i < ECL_GetCard(currentPath); i++)
        {
            substructInstanceIdxs[params.pathSegmentIdx] = i;
            
            if (params.pathSegmentIdx < params.substructPathSegments.length - 2)
            {
                substructuresLoop ({
                    substructPathSegments: params.substructPathSegments,
                    currentPath: currentPath + '.RANGE.' + i,
                    pathSegmentIdx: params.pathSegmentIdx + 1
                });
            }
            else
            {
                var exactFullPath = currentPath + '.RANGE.' + substructInstanceIdxs[params.pathSegmentIdx] + '.' + params.substructPathSegments[params.pathSegmentIdx+1];
                var generalFullPath = currentPath + '.' + params.substructPathSegments[params.pathSegmentIdx+1];
                
                if (substructConditionResults[exactFullPath] == true || substructConditionResults[generalFullPath] == true)
                {
                    conditionsMet[curConditionIdx].push(exactFullPath);
                }
                else
                {
                    conditionsNotMet[curConditionIdx].push(exactFullPath);
                }
            }
        }
    };
    
    for (curConditionIdx = 0; curConditionIdx < actionParams.substructCondPaths.length; curConditionIdx++)
    {
        conditionsMet[curConditionIdx] = [];
        
        substructuresLoop({
            substructPathSegments: actionParams.substructCondPaths[curConditionIdx].split('.'),
            currentPath: '',
            pathSegmentIdx: 0
        });
    }
    
    if (actionParams.ruleMet)
    {
        for (var i = 0; i < conditionsMet.length; i++)
        {
            numExecutions = numExecutions * conditionsMet[i].length;
        }
    }
    else
    {
        for (var i = 0; i < conditionsNotMet.length; i++)
        {
            numExecutions = numExecutions * conditionsNotMet[i].length;
        }
    }
    
    for (var i = 0; i < numExecutions; i++)
    {
        eval(actionParams.action);
    }
}


// Iterate over passed datasets defined by dynamic rules tables, retrieve name of variables referencing strings set
// via ECL function 'ECL_SetInfoVariable' and resolve them (i.e. replace variable name with referenced string)
//
//	@param datasets: Array of dataset to process. Each dataset must be an object
//	@param infoVarProperty: Name of property of each dataset containing string to search for variable names
function replaceInfoVariables (params)
{
    if (params != undefined && params.datasets != undefined && params.infoVarProperty != undefined)
    {
        for (var i = 0; i < params.datasets.length; i++)
        {
            if (typeof params.datasets[i][params.infoVarProperty] != 'string')
            {
                continue;
            }
            
            var infoVariablesContained = params.datasets[i][params.infoVarProperty].match(/\(\$[^\$\)]+\$\)/g);
            if (infoVariablesContained != null)
            {
                var updatedString = params.datasets[i][params.infoVarProperty];
                for (var j = 0; j < infoVariablesContained.length; j++)
                {
                    updatedString = updatedString.replace(infoVariablesContained[j], ECL_GetInfoVariable(infoVariablesContained[j].slice(2, -2)));
                }
                params.datasets[i][params.infoVarProperty] = updatedString
            }
        }
    }
}



// Iterate over all BOM datasets defined by dynamic rules tables, determine those whose article number occurs multiple
// times, and cumulate them, if desired via respective flag. Cumulating of datasets with same article number means:
// - summation of quantity defined by each dataset
// - calculation of overall unit price by at first multiplying price defined by each dataset with its quantity and
//	 afterwards division of result by summed up quantity of all datasets
// - overwriting of quantity and unit price of first dataset (according to order of BOM) and deletion of all further
//	 datasets
function cumulateBomQuantities (dynBomDatasets)
{
    if (dynBomDatasets.length > 1)
    {
        var qtyCumulateHandling = {};
        var cumulatedBomDatasets = [];
        
        for (var i = 0; i < dynBomDatasets.length; i++)
        {
            var curDataset = dynBomDatasets[i];
            var cumulateQty = curDataset.cumulateQty;
            
            delete curDataset.cumulateQty;
            
            if (cumulateQty)
            {
                if (qtyCumulateHandling[curDataset.sArticle] == undefined)
                {
                    cumulatedBomDatasets.push(curDataset);
                    
                    qtyCumulateHandling[curDataset.sArticle] = {
                        referenceDataset: curDataset,
                        cumulateDatasets: [curDataset]
                    };
                }
                else
                {
                    qtyCumulateHandling[curDataset.sArticle].cumulateDatasets.push(curDataset);
                }
            }
            else
            {
                cumulatedBomDatasets.push(curDataset);
            }
        }
        
        for (var artNr in qtyCumulateHandling)
        {
            var cumulatedPrice = 0;
            var cumulatedQty = 0;
            var cumulateDatasets = qtyCumulateHandling[artNr].cumulateDatasets;
            
            for (var i = 0; i < cumulateDatasets.length; i++)
            {
                cumulatedQty += cumulateDatasets[i].nQuantity;
                cumulatedPrice += parseFloat(cumulateDatasets[i].dPrice * cumulateDatasets[i].nQuantity);
            }
            
            qtyCumulateHandling[artNr].referenceDataset.dPrice = parseFloat(cumulatedPrice / cumulatedQty).toFixed(2);
            qtyCumulateHandling[artNr].referenceDataset.nQuantity = cumulatedQty;
        }
        
        return cumulatedBomDatasets;
    }
    else
    {
        return dynBomDatasets;
    }
}